/*
 *      unixinout.c -- input sound samples, output to konsole
 *
 *      Copyright (C) 1996  
 *          Thomas Sailer (sailer@ife.ee.ethz.ch, hb9jnx@hb9w.che.eu)
 *
 *      Copyright (C) 1998-2003
 *          Markus Grohmann (markus_grohmann@gmx.de)
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * ---------------------------------------------------------------------- */

//#define ZLOG

#define ZEILEN

#include "monitor.h"
#include <stdio.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <time.h>

#ifdef SUN_AUDIO
#include <sys/audioio.h>
#include <stropts.h>
#include <sys/conf.h>
#else /* SUN_AUDIO */
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#endif /* SUN_AUDIO */

/* ------------------------------------------------------------------- */

static const char *allowed_types[] = {
	"raw", "aiff", "au", "hcom", "sf", "voc", "cdr", "dat", 
	"smp", "wav", "maud", "vwe", NULL
};

/* ------------------------------------------------------------------- */

static const struct	demod_param *dem[] = { ALL_DEMOD };

#define NUMDEMOD		(sizeof(dem)/sizeof(dem[0]))

static struct				demod_state dem_st[CHANNELS * NUMDEMOD];
static unsigned int	dem_mask[2][(NUMDEMOD + 31) / 32 * CHANNELS];

int				errflg = 0;
unsigned int	overlap = 0, mask_first[2][CHANNELS];

#define MASK_SET(l, m, n)	 	dem_mask[l][m] |=   1 << ((n) & 0x1f)
#define MASK_RESET(l, m, n)	dem_mask[l][m] &= ~(1 << ((n) & 0x1f))
#define MASK_ISSET(l, m, n)	dem_mask[l][m] &    1 << ((n) & 0x1f)

#define END_LIST(l, v)			l - v

#define	USE	0
#define	HELP	1

/* ------------------------------------------------------------------- */

static const char usage_str[] = "Monitor demoduliert und dekodiert viele verschiedene Funkbertragungsformate.\r\n"
"\r\n"
"  -t <type> <file>     : Eingabe Dateityp (andere Typen als raw bentigen sox)\r\n"
"  -a -{L|R} <modul>    : addiere Modul\r\n"
"  -s -{L|R} <modul|all>: subtrahiere Modul\r\n"
"  -f <rc-file>         : Resourcendatei\r\n"
"  -h                   : Hilfe\r\n";

/*-------------------------------------------------------------------- */

static void error (void) {
	int i;

	if (!errflg) return;
//	endwin();
	fprintf(stdout, "\rMonitor (C) 1998-2003 by Markus Grohmann\r\nverfgbare Demodulatoren:\r\n");
	for (i = 0; i < NUMDEMOD; i++)
		fprintf(stdout, " %s", dem[i]->name);
	fprintf(stdout, "\r\n\n");
	(void) fprintf(stdout, usage_str);
	exit(2);
}	/*	error

---------------------------------------------------------------------- */

void zeilen(int zeile) {
	/*	Zeilenspeicher	*/

	#ifdef ZEILEN
	WINDOW	*vwin = common.windows[VERL];
	char		counter[ITEMLEN];

	sprintf(counter, "    %i/%i", zeile, len_list);
	wattrset(vwin, common.color[COL_NEUTRAL]);
	mvwaddstr(vwin, vwin->_maxy, vwin->_maxx - strlen(counter), counter);
	#endif
}	/*

---------------------------------------------------------------------- */

void update_file(char *outprint, unsigned short log) {
/*****************************************************************************	
 *	Logfile schreiben
 *****************************************************************************/

	char	ss[ITEMLEN], logfile[NAMELEN], outlog[LINELEN];
	int	i;

	if (!log || !*common.logfile) return;
	strcpy(logfile, common.logfile);
	strncat(logfile, get_date(ss), 10);
	strcat(logfile, ".log");

	sprintf(outlog, "%s\n", outprint);

	/*	ffnen der Logdatei	*/
	if ((i = open(logfile, O_WRONLY)) < 0) {

		/*	Datei nicht vorhanden, erzeugen einer Datei	*/
		i = creat(logfile, S_IWRITE + S_IREAD + S_IROTH + S_IRGRP);
		write(i, outlog, strlen(outlog));
		close(i);

		/*	Neuffnen der erzeugten Datei	*/
		if ((i = open(logfile, O_WRONLY)) < 0) {

			/*	Datei nicht vorhanden, Anlegen des Verzeichnisses	*/
			char path[NAMELEN];
			memset(path, 0, sizeof(path));
			strcpy(path, "mkdir ");
			strcat(path, logfile);
			*strrchr(path, '/') = 0;
			system(path);

			/*	Erzeugen der Datei	*/
			i = creat(logfile, S_IWRITE + S_IREAD + S_IROTH + S_IRGRP);
			write(i, outlog, strlen(outlog));
			close(i);

			/*	Datei tatschlich vorhanden?	*/
			if ((i = open(logfile, O_WRONLY)) < 0) {
				*statout = 0; verbprintf(ALWAYS, "");
				perror("open");
			}
		}
	}
	else {

		/*	Springe ans Ende der Datei und fge Eintrag an	*/
		lseek(i, 0, SEEK_END);
		write(i, outlog, strlen(outlog));
	}
	close(i);
}	/*	update_file

---------------------------------------------------------------------------- */

void verbprintf(int verb_level, const char *fmt, ...) {
/*****************************************************************************	
 *	Statusanzeigen ausgeben
 *****************************************************************************/

	WINDOW	*win = common.windows[VERL];
	char		outprint[LINELEN];
	int		i, max_letters = win->_maxx, max_lines = win->_maxy;
	va_list	args;
 
	if (verb_level > verbose) return;
	
	va_start(args, fmt);
   vsprintf(outprint, fmt, args);
	va_end(args);

	for (i = strlen(outprint); i <= max_letters; i++) strncat(outprint, " ", 1);
	wattrset(win, common.color[COL_NEUTRAL]);
	mvwaddstr(win, max_lines, 0, outprint);
	wrefresh(win);
}	/*	verbprintf

---------------------------------------------------------------------------- */

void Sprlist(void) {
/*****************************************************************************	
 *	Sprechwnsche anzeigen
 *****************************************************************************/

	Sprwunsch	*akt = common.sprechwunsch;
	int			i = 1;

	for (;i <= sprwlines; i++) {
		/*	Farbe	*/
		if (akt != NULL && akt->not) wattrset(common.windows[SPRW], common.color[15]);
		else {
			if (i == 1) wattrset(common.windows[SPRW], common.color[12]);
			else wattrset(common.windows[SPRW], common.color[13]);
		}

		/*	Text	*/
		if (akt == NULL) mvwaddstr(common.windows[SPRW], i, 0, "                ");
		else {
			mvwaddstr(common.windows[SPRW], i, 0, akt->who);
			akt = akt->next;
		}
	}
}	/*	Sprlist

---------------------------------------------------------------------------- */

void Sprwin(void) {
/*****************************************************************************	
 *	Sprechwunschfenster frben, beschriften
 *****************************************************************************/

	WINDOW	**win = common.windows;
	int		_rows = sprwlines,
				x, y, leg = common.legende;

	_rows = (_rows + 1 < leg ? _rows + 2 : leg - 1);
	_rows = (_rows  < 4 ? leg + 1 : _rows);
	if (leg >= _rows) {
		win[SPRW] = derwin(win[MAIN], _rows - 1, 16, leg - _rows, win[MAIN]->_maxx - 17);
		wattrset(win[SPRW], 
			(common.color[11] & A_UNDERLINE
				? common.color[11] ^ A_UNDERLINE
				: common.color[11]));
		for (y = 0; y < _rows; y++) {
			for (x = 0; x < 16; x++) {
				mvwaddstr(win[SPRW], y, x, " ");
			}
		}
		wattrset(win[SPRW], common.color[11]);
		mvwaddstr(win[SPRW], 0, 1, LOCAL_TEXT(" Sprechwunsch "));
		Sprlist();
		wrefresh(win[SPRW]);
	}
}	/*	Sprwin

--------------------------------------------------------------------------- */

char * update_view(
	char			*outprint,
	int				color,	/*	Farbe	*/
	unsigned short shift,	/*	n-te Zeile von oben	*/
	unsigned short bit		/*	Einfgen oder ersetzen,	Zeile in Log	*/
	) {
/**************************************************************************
 * 
 **************************************************************************/

#define MAXMEM	50000
	WINDOW	*win = common.windows[VERL];
#define SPACE	win->_maxx + 1
	Line	*line = common.line, *pre = NULL, *mem = NULL, **root;
	int	i = 0, max_lines = win->_maxy;
	char	clean[LINELEN];

	line = common.line;
	root = &common.line;
	update_file(outprint, bit & LOG);
	memset(clean, ' ', sizeof(clean));

	i = shift;
	mem = line;
	while (i-- && line != NULL) {
	   pre = mem;
		mem = mem->next;
		line = line->next;
	}
	if (NEW) {
		line = (Line *) malloc (sizeof(Line));
		memset(line, 0, sizeof(Line));
		if (len_list < MAXMEM) len_list++;
		if (shift) pre->next = line;
		else *root = line;
		line->next = mem;
	}
	memset(line->string, 0, sizeof(clean));
	if (strlen(outprint) > SPACE) strncpy(line->string, outprint, SPACE);
	else {
		strncpy(line->string, outprint, strlen(outprint));
		strncat(line->string, clean, SPACE - strlen(outprint));
	}
	line->value = color;
	common.line = *root;

	line = *root;
	i = 0;
	/*	Schreibe den Anfang des Zeilenpuffers ins Verlauffenster	*/
	while (line->next != NULL && i < max_lines - 1) {
		wattrset(win, common.color[line->value]);
		mvwaddstr(win, i, 0, line->string);
		i++;
		line = line->next;
	}
	wattrset(win, common.color[line->value]);
	mvwaddstr(win, i, 0, line->string);

	/*	Weiter durch den Puffer, bis maximale Gre erreicht	*/
	while (line->next != NULL && i++ < MAXMEM) line = line->next;
	/*	Rest lschen	*/
	if (i >= MAXMEM && line->next != NULL)	clear_line(&line->next);
	if (line->next != NULL)	clear_line(&line->next);

	zeilen(1);
	wrefresh(win);

	return(NULL);
}	/*	update_view

---------------------------------------------------------------------------- */

void solve_format(char *line, Format **format, const char *Tokens[], int numtokens) {
/*****************************************************************************	
 *	Einlesen von Formatierungszeilen
 *****************************************************************************/

	Format	*root = NULL;
	char	*ss, *wht = " ";
	unsigned int	i, j, first = 1, is_root = 1, chrs = 0, len = 0;

	if (*line == 0) {
		verbprintf(ALWAYS, "%s: Formatierung fehlt oder nicht korrekt.", *Tokens);
		return;
	}
	ss = line;
	while (*ss) {

		i = 1;
		j = -1;
		while ((i < numtokens) && (j = strncmp(ss, Tokens[i], strlen(Tokens[i]))))
			i++;

		if (!j) {
			ss += strlen(Tokens[i]);
			if (isDezimal(*ss)) {
				len = str2dez(&ss);
				if (!len) {
					verbprintf(ALWAYS, "%s: Lngenangabe muss grer Null sein: %s%s", *Tokens, Tokens[i], ss);
					return;
				}
				if (len >= 100) {
					verbprintf(ALWAYS, "%s: Lngenangabe ein- oder zweistellig: %s%d%s", *Tokens, Tokens[i], len, ss);
					return;
				}
			}
			else {
				verbprintf(ALWAYS, "%s: Lngenangabe fehlt.", *Tokens);
				return;
			}
			if (is_root) {
				(*format) = (Format *) malloc (sizeof(Format));
				root = *format;
				is_root = 0;
			}
			else {
				(*format)->next = (Format *) malloc (sizeof(Format));
				(*format) = (*format)->next;
			}
			memset(*format, 0, sizeof(Format));
			(*format)->choice.wich = i - 1;
			(*format)->len = len;

			first = 1;
		}

		else	{
			if (first) {
				chrs = 0;
				first = 0;
				if (is_root) {
					(*format) = (Format *) malloc (sizeof(Format));
					root = *format;
					is_root = 0;
				}
				else {
					(*format)->next = (Format *) malloc (sizeof(Format));
					(*format) = (*format)->next;
				}
				memset(*format, 0, sizeof(Format));
				wht = (*format)->choice.what;
				(*format)->len = 0;
			}			
			if (chrs < ITEMLEN) {
				chrs++;
				*wht++ = *ss++;
			}
			else {
				chrs = 0;
				first = 1;
			}
		}
	}
	(*format)->next = NULL;
	*format = root;
}	/*	solve_format

* ---------------------------------------------------------------------- */

void solve_txt_sub(char *line, Line **sub) {
/*****************************************************************************	
 *	Einlesen von Textersatzregeln
 *****************************************************************************/

	char	*st, str[2][ITEMLEN], *ss = line;
	int	i, k;
	Line	*akt = NULL, *root = *sub;

	for (i = 0; i < 2; i++) {

		go(&ss);

		if (*ss++ != 34) {
			verbprintf(ALWAYS, "SUBST: \" oder \\# erwartet: %s", line);
			return;
		}
		st = ss;
		k = 0;
		while (*st != 34 && *st != 0) {
			/*	Ende suchen	*/
			k++;
			st++;
		}
		if (*st != 34) {
			verbprintf(ALWAYS, "SUBST: \" oder \\# erwartet: %s", line);
			return;
		}
		memset(str[i], 0, ITEMLEN);
		strncpy(str[i], ss, k);
		ss += (k + 1);
	}
	while (akt != NULL) {
		akt = akt->next;
	}
	akt = (Line *) malloc (sizeof(Line));
	memset(akt, 0, sizeof(Line));
	strcpy(akt->string, str[0]);
	strcpy(akt->subst,  str[1]);
	*sub = akt;
	if (root != NULL) (*sub)->next = root;

}	/*	solve_subst

---------------------------------------------------------------------- */

void monrc(struct demod_state *s, Format **format, char *filename, int modul) {

int	fd, er;
char	*ss, line[LINELEN], byte;

	void solve_line(void) {

		extern void pocsag_line(struct demod_state *s, Format **format, char *line);

		switch (modul) {
		 case -1:
		 	pocsag_line(s, format, line);
		 	break;

		 default:
		 	dem[modul]->line(s, format, line);
		}
	}

	if ((fd = open(filename, O_RDONLY)) < 0) {
		verbprintf(ALWAYS, "%s kann nicht geffnet werden", filename);
		exit(1);
	}

	ss = line;
	memset(line, 0, sizeof(line));

	while (read(fd, &byte, 1)) {

		if (byte != 10 && byte != 13) {

			if (byte == '#') {
				if (*(ss - 1) == '\\') {
					*(ss - 1) = '#';
				}
				else {
					/*	alles rechts von # wird ignoriert	*/
					while ((er = read(fd, &byte, 1)) && byte != 10 && byte != 13);
					solve_line(/*s, format, line*/);
					memset(line, 0, sizeof(line));
					ss = line;
					if (!er) break;
				}
			}
			else *ss++ = byte;
		}
		else {
			/*	Auswertung von Line	*/
			solve_line(/*s, format, line*/);
			memset(line, 0, sizeof(line));
			ss = line;
		}

	}
	
	if (ss != &(line[0])) {
		*ss = 0;
		/*	Auswertung der letzten Line	*/
		solve_line(/*s, format, line*/);
	}
	close(fd);
}	/*	monrc

 ---------------------------------------------------------------------- */

void InitColors(void) {
/***********************************************************************	
 *	Einlesen der Farben
 ***********************************************************************/

	char		*ss, line[LINELEN];
	char		*rcfile = common.rcfile.name;
	int		fd, er = 1, i, col[3];
	long int	*color = common.color;

	start_color();

	if ((fd = open(rcfile, O_RDONLY)) < 0) {
		printf("\n%s kann nicht geffnet werden", rcfile);
		exit(1);
	}

	ss = line;

	while (read(fd, ss, 1)) {

		if (*ss == 10 || *ss == 13)
			ss = line;
		else {

			if (*ss == '#') {
				do er = read(fd, ss, 1);
				while (er && (*ss != 10 || *ss == 13));
					ss = line;
			}
			else {
				if (!strncmp(line, "COLOR", 5)) {
					ss = line;

					do er = read(fd, ss, 1);
					while (er && (*ss++ != 10 || *ss == 13));

					ss = line;
				
					for (i = 0; i < 3; i++) {

						go(&ss);

						if (isDezimal(*ss))
							col[i] = str2dez(&ss);
						else
							col[i] = -1;

						if (i == 0) {
							if (col[0] < 0 || col[0] > 50) {
								printf("\nFehler in COLOR-Nummer, Werte 0 bis 50.\n");
								exit(15);	/*	Programmabbruch */
							}
							if (col[0] == 15) {
								printf("\nCOLOR 15 ist reserviert\n");
								exit(15);	/*	Programmabbruch */
							}
							if (col[0] == 0) col[0] = COL_NOTRUF;
						}
						else {
							if (col[i] < 0) {
								printf("\nFehler in COLOR-Angabe, Wert fehlt.\n");
								exit(15);	/*	Programmabbruch */
							}
						}
					}

					init_pair(col[0], col[1], col[2]);
					color[col[0]] = COLOR_PAIR(col[0]);

					while (*ss != 10 && *ss != 13 && (*ss == 9 || *ss == ' ')) ss++;
					
					while (*ss == 'U' || *ss == 'B' || *ss == 'F' || *ss == 'R') {
						if (*ss == 'B') {color[col[0]] |= A_BLINK; ss++;}
						if (*ss == 'F') {color[col[0]] |= A_BOLD; ss++;}
						if (*ss == 'R') {color[col[0]] |= A_REVERSE; ss++;}
						if (*ss == 'U') {color[col[0]] |= A_UNDERLINE; ss++;}
					}

					ss = line;
					ss--;
				}
				if (!er) break;
				ss++;
			}
		}
	}
	close(fd);
}	/*	InitColors

---------------------------------------------------------------------- */

void ClearLayout(void) {
/***********************************************************************	
 *	Layout lschen
 ***********************************************************************/
	int		x, y = 0, leg = common.legende;
	WINDOW	*win = common.windows[MAIN];

	/*	Layout-Fenster	*/
	wattrset(win, common.color[10]);
	for (; y < leg; y++) {
		for (x = 0; x < win->_maxx; x++) {
			mvwaddstr(win, y, x, " ");
		}
	}
	wrefresh(win);
}	/*	ClearLayout

---------------------------------------------------------------------- */

void InitScreen(void) {
/***********************************************************************	
 *	Layout anlegen
 ***********************************************************************/

	WINDOW	**win = common.windows, *time_win, *vwin;
	int
		i = 0, y = 0, leg, wl, wc;
	char	hz[ITEMLEN];

	if (common.scn.lines)
		wl = common.scn.lines;
	else {
		wl = common.scn.lines = LINES;
		getch(); /*	ntig fr Display	*/
	}

	if (common.scn.cols)
		wc = common.scn.cols;
	else
		wc = common.scn.cols = COLS;


	/*	ggf. vor Neuanlegen alte Fenster lschen	*/
	for (i = 4; i; i--) {
		if (win[i] != NULL) {
			y = delwin(win[i]);
			win[i] = NULL;
		}
	}

	/*	Hauptfenster	*/
	if (win[ROOT] == NULL) win[ROOT] = newwin(wl - 1, wc, 0, 0);
 	win[ROOT]->_maxx = wc;
 	win[ROOT]->_maxy = wl - 1;
 
	/*	Hauptfenster	*/
	win[MAIN] = derwin(win[ROOT], win[ROOT]->_maxy, win[ROOT]->_maxx, 0, 0);

	leg = MIN(common.legende, win[MAIN]->_maxy - 2);
	common.legende = leg;

	/*	Verlaufsfenster	*/
	win[VERL] = derwin(win[MAIN], win[MAIN]->_maxy - leg, win[MAIN]->_maxx, leg + 1, 0);

	/*	Legende	*/
	win[LGND] = derwin(win[MAIN], 1, win[MAIN]->_maxx, leg, 0);

	time_win = common.windows[LGND];
	mvwaddstr(time_win, 0, 0, "bitte warten...  ");

	verbprintf(ALWAYS, " ");

	sprintf(hz, "%uHz", FREQ_SAMP);
	vwin = common.windows[VERL];
	wattrset(vwin, common.color[COL_NEUTRAL]);
	mvwaddstr(vwin, vwin->_maxy, vwin->_maxx - strlen(hz), hz);

	common.scn.cnt = 5;
	ClearLayout();
}	/*	InitScreen

---------------------------------------------------------------------- */

int windowResized (void) {
/***********************************************************************	
 *	Fenstergre abfragen, fhrt noch zu Abstrzen
 ***********************************************************************/

#include <signal.h>

	return FALSE; /*	immer	*/

	if (!common.scn.cnt--) {
		struct winsize ws;
		ioctl(1, TIOCGWINSZ, &ws);

		common.scn.cnt = 5;
		if (common.scn.lines > ws.ws_row || common.scn.cols > ws.ws_col) {
			common.scn.lines = ws.ws_row;
			common.scn.cols  = ws.ws_col;
			return TRUE;
		}
	}
	return FALSE;
}	/* windowResized

---------------------------------------------------------------------- */

void solve_line_main(char *line) {
/***********************************************************************	
 *	Entdecken relelevanter Zeilen in der monrc
 ***********************************************************************/

	const char	*Type[3]= {"MOD", "LEGENDE", "LOGPATH"};

	char	*ss = line;
	int	i = 0, j = -1, chan;

	while((i < 3) && (j = strncmp(ss, Type[i], strlen(Type[i]))))
		i++;

	if (j)
		return;

	ss += strlen(Type[i]) + 1;
	while (*ss == 9) ss++;

	switch (i) {

	case 0:	/*	MOD	*/
		while (*ss == 9 || *ss == ' ') ss++;
		switch (*ss++) {

		case 'L':
			chan = 0;
			break;

		case 'R':
			chan = 1;
			break;

		case 'M':
			chan = 2;
			break;

		default:
			printf("\nMOD: Kanal 'L' oder 'R' angeben.\n");
			exit(1);	
		}

		go(&ss);

		for (i = 0; i < NUMDEMOD; i++)
			if (!strncasecmp(ss, dem[i]->name, strlen(dem[i]->name))) {
				if (mask_first[HELP][chan]) {
					mask_first[HELP][chan] = 0;
					dem_mask[HELP][chan] = 0;
					/*	hier wird gleich noch der andere Kanal gelscht	*/
					j = (chan + 1) % 2;
					if (mask_first[j]) {
						mask_first[HELP][j] = 0;
						dem_mask[HELP][j] = 0;
					}
				}
				MASK_SET(HELP, chan, i);
				break;
			}
		if (i >= NUMDEMOD && !errflg) {
			fprintf(stderr, "Unbekanntes Modul \"%s\"\n\n", ss);
			errflg++;
		}
		break;

	case 1:	/*	LEGENDE	*/
		common.legende = str2dez(&ss);
		break;

	case 2:	/*	LOGPATH	*/
		strcpy(common.logfile, getenv("HOME"));
		strcat(common.logfile, "/");
		while (*ss && *ss != 10 && *ss != 13) strncat(common.logfile, ss++, 1);
		strcat(common.logfile, "/log_");
		break;
	}
}	/* solve_line_main

---------------------------------------------------------------------- */

void monrc_main(void) {
/***********************************************************************	
 *	Anlegen einer monrc (falls nicht vorhanden),
 *	Einlesen globaler Informationen
 ***********************************************************************/

	int	fd, fd0, er;
	char	*ss, line[LINELEN], byte;
   struct stat crt;

	ss = "##################################################################################\n"
"#\n"
"# Modes (MOD Kanal Modul)\n"
"#\n"
"MOD	L	fms\n"
"MOD	L	ZVEI\n"
"MOD	L	DTMF\n"
"MOD	L	POCSAG512\n"
"#MOD	R	pocsag1200\n"
"#\n"
"#Formatierung fr FMS/ZVEI/POCSAG-Ausgabe (NAME Lnge Fllzeichen)\n"
"#\n"
"#FMS: ZEIT8 KANALnnBOSnn LANDnn ORTnn KFZnn STATUSnn BSTnn TKInn DIRnn\n"
"FMS:ZEIT8 KANAL1 BOS4 LAND3 ORT3 KFZ10STATUS30TKI3 Dir:DIR1 Bst:BST1\n"
"\n"
"#ZVEI: ZEIT8 KANALnn NAMEnn TYPEnn NUMMERnn\n"
"ZVEI:ZEIT8 KANAL1 NAME22 TYPE15 NUMMER5\n"
"\n"
"#POCSAG: ZEIT8 KANALnn NAMEnn NUMMERnn FUNCn\n"
"POCSAG:ZEIT8 KANAL1 NAME22 NUMMER7 Type: FUNC1 Pager\n"
"\n"
"LEGENDE	0			#Hhe der Datumslinie von oben\n"
"SPRWLINES	2		#Anzahl Zeilen des Sprechwunschfensters\n"
"PTT	11,4,8,9		#Formatierung fr Sprechwunschfenster\n"
"QUITTUNG 1			#Quittung FMS: 0-aus, 1-ein, 2-nur, falls kein Status verstanden wurde\n"
"SYNCBITS	8			#Lnge des Syncworts fr FMS (8)\n"
"CRC		1			#FMS-Fehlercheck\n"
"STEUERZEICHEN	1	#fr Pocsag und FMS-Texte\n"
"SQUELCHFMS	1		#absoluter Mindestpegel fr FMS\n"
"DUMP	0				#Hexdatenanzeige fr FMS, Pausenzeiten fr ZVEI\n"
"CORRPOC	0\n"
"\n"
"STATUS_F_KFZ	0	0-Notruf\n"
"STATUS_F_KFZ	1	1-frei ber Funk\n"
"STATUS_F_KFZ	2	2-einsatzbereit auf Wache\n"
"STATUS_F_KFZ	3	3-Einsatz bernommen\n"
"STATUS_F_KFZ	4	4-am Einsatzort\n"
"STATUS_F_KFZ	5	/C11	5-Sprechwunsch\n"
"STATUS_F_KFZ	6	6-nicht einsatzbereit\n"
"STATUS_F_KFZ	7	7-Patient aufgenommen\n"
"STATUS_F_KFZ	8	8-im Krankenhaus\n"
"STATUS_F_KFZ	9	9-Anmeldung\n"
"STATUS_F_KFZ	a	Standort\n"
"STATUS_F_KFZ	b	$$B (Kfz)\n"
"STATUS_F_KFZ	c	$$C (Kfz)\n"
"STATUS_F_KFZ	d	$$D (Kfz)\n"
"STATUS_F_KFZ	e	Fahrzeugquittung\n"
"STATUS_F_KFZ	f	Sprechtaste\n"
"\n"
"STATUS_F_LST	0	Statusabfrage\n"
"STATUS_F_LST	1	A-Sammelruf\n"
"STATUS_F_LST	2	E-Einrcken/Abbrechen\n"
"STATUS_F_LST	3	/C5	C-fr Einsatzbernahme melden\n"
"STATUS_F_LST	4	F-ber Telefon melden\n"
"STATUS_F_LST	5	H-Wache anfahren\n"
"STATUS_F_LST	6	J-Sprechaufforderung\n"
"STATUS_F_LST	7	L-Lagemeldung durchgeben\n"
"STATUS_F_LST	8	P-Fernwirken\n"
"STATUS_F_LST	9	U-Fernwirken II\n"
"STATUS_F_LST	a	Textbertragung\n"
"STATUS_F_LST	b	Zielort durchgeben\n"
"STATUS_F_LST	c	$$C (Lst)\n"
"STATUS_F_LST	d	$$D (Lst)\n"
"STATUS_F_LST	e	Bereitschaftsanfrage\n"
"STATUS_F_LST	f	automatische Quittung\n"
"\n"
"STATUS_P_KFZ	0	0-Notruf\n"
"STATUS_P_KFZ	1	1-frei auf Streife\n"
"STATUS_P_KFZ	2	2-einsatzbereit auf Wache\n"
"STATUS_P_KFZ	3	3-Einsatz bernommen\n"
"STATUS_P_KFZ	4	4-am Einsatzort\n"
"STATUS_P_KFZ	5	/C11	5-Sprechwunsch\n"
"STATUS_P_KFZ	6	6-nicht einsatzbereit\n"
"STATUS_P_KFZ	7	$$\n"
"STATUS_P_KFZ	8	$$\n"
"STATUS_P_KFZ	9	9-Fremdanmeldung\n"
"STATUS_P_KFZ	a	$$\n"
"STATUS_P_KFZ	b	$$\n"
"STATUS_P_KFZ	c	$$\n"
"STATUS_P_KFZ	d	$$\n"
"STATUS_P_KFZ	e	Fahrzeugquittung\n"
"STATUS_P_KFZ	f	Sprechtaste\n"
"\n"
"STATUS_P_LST	0	Statusabfrage\n"
"STATUS_P_LST	1	A-Sammelruf\n"
"STATUS_P_LST	2	E-Eingesicherung beachten\n"
"STATUS_P_LST	3	/C5	C-fr Einsatzbernahme melden\n"
"STATUS_P_LST	4	F-ber Telefon melden\n"
"STATUS_P_LST	5	H-Dienststelle anfahren\n"
"STATUS_P_LST	6	J-Sprechaufforderung\n"
"STATUS_P_LST	7	L-Einsatz abbrechen\n"
"STATUS_P_LST	8	P-Fernwirken I\n"
"STATUS_P_LST	9	U-Fernwirken II\n"
"STATUS_P_LST	a	$$\n"
"STATUS_P_LST	b	$$\n"
"STATUS_P_LST	c	$$\n"
"STATUS_P_LST	d	$$\n"
"STATUS_P_LST	e	$$\n"
"STATUS_P_LST	f	automatische Quittung\n"
"\n"
"COLOR	1	2	7		#Status1\n"
"COLOR	2	0	7		#Status2\n"
"COLOR	3	7	1		#Status3\n"
"COLOR	4	0	3		#Status4\n"
"COLOR	5	5	7	B	#Alarmierung\n"
"COLOR	6	7	0		#Status6\n"
"COLOR	7	7	4		#Status7\n"
"COLOR	8	0	6		#Status8\n"
"COLOR	9	4	7		#Status9\n"
"COLOR	10	0	7		#Initialisierung und Layouthintergrund\n"
"COLOR	11	7	4	FU	#Sprechwunsch-Fenster und berschrift\n"
"COLOR	12	7	4	B	#Sprechwunsch-Fenster 1. Zeile\n"
"COLOR	13	7	4		#Sprechwunsch-Fenster 2. Zeile\n"
"\0";

	if ((fd = open(common.rcfile.name, O_RDONLY)) < 0) {
		fd = creat(common.rcfile.name, S_IWRITE + S_IREAD + S_IROTH + S_IRGRP);
		write(fd, ss, strlen(ss));
		close(fd);

		if ((fd = open(common.rcfile.name, O_RDONLY)) < 0) {
			printf("\n%s wurde erstellt, %s kann nicht geffnet werden", common.rcfile.name, common.rcfile.name);
			exit(1);
		}
	}

	ss = line;

	while (read(fd, &byte, 1)) {

		if (byte != 10 && byte != 10){

			if (byte == 35) {
				/*	alles rechts von # wird ignoriert	*/
				while((er = read(fd, &byte, 1)) && byte != 10 && byte != 13);
				*ss = 0;
				solve_line_main(line);
				ss = line;
				if (!er) break;
			}
			else *ss++ = byte;
		}
		else {
			*ss = 0;

			/*	Auswertung von Line	*/
			solve_line_main(line);
			ss = line;
		}
	}
	
	if (ss != &(line[0])) {
			*ss = 0;

		/*	Auswertung der letzten Line	*/
		solve_line_main(line);		
	}
	close(fd);

	/* Wann wurde das rc-File erstellt? */
	if (!common.rcfile.create_time) {
	   fd0 = stat(common.rcfile.name, &crt);
		
		if(fd0) {
			/*	falls da nichts kommt,
			 * nehmen wir einfach die Zeit des Programmstarts	*/
			time(&tp);
			common.rcfile.create_time = tp;
		}
		else common.rcfile.create_time = crt.st_mtime;
	}
}	/* monrc_main

---------------------------------------------------------------------- */

void init_struct (int ichan, int modul) {
/***********************************************************************
 *	Initialisieren der Datenstruktur fr ein bentigtes Modul
 ***********************************************************************/

	int				i = modul;
	unsigned int	shift;

	/*	Datenstruktur initialisieren	*/
	shift = ichan * NUMDEMOD + i;

	dem_st[shift].dem_par = dem[i];
	if (dem[i]->init)
		dem[i]->init(dem_st + shift);

	first_init = 0;
	if (dem[i]->overlap > overlap)
		overlap = dem[i]->overlap;
}	/* init_struct

---------------------------------------------------------------------- */

void initall(void) {
/***********************************************************************	
 *	Initialisieren von diesem und jenem
 ***********************************************************************/

	int ichan;

	/*	Setzen von mask_first[HELP] auf allen Kanlen auf 1.
	 *	in mask_first[USE] stehen beim ersten Mal diejenigen Module,
	 *	die als Argumente angegeben wurden.
	 *	Bei Neuinit stehen da die laufenden Module drin	*/
	for (ichan = 0; ichan < CHANNELS; ichan++) {
		mask_first[HELP][ichan] = 1;
		dem_mask[HELP][ichan] = 0;
	}

	error();

	/*	u.a. holen der Module aus der .monrc,
	 *	Setzen der entsprechenden Bits in dem_mask[HELP]	*/
	monrc_main();
	if (!(dem_mask[HELP][0]) && !(dem_mask[HELP][1]))
		for (ichan = 0; ichan < CHANNELS; ichan++) {
			memset(dem_mask[HELP] + ichan, 0xff, sizeof(dem_mask[HELP][ichan]));
	}
	
	error();

	strcpy(common.layoutfile, getenv("HOME"));
	strcat(common.layoutfile, "/.layout");

	initscr();
	leaveok(stdscr, TRUE);
	keypad(stdscr, TRUE);    
	timeout(0);

	InitColors();
	InitScreen();
}	/*	initall

---------------------------------------------------------------------- */

void process_buffer(float *buf, unsigned int len) {
/***********************************************************************	
 *	Aufruf aller aktivierten Module mit den eingelesenen Sounddaten
 ***********************************************************************/

	int			i, chkdate = TRUE, shift = 0;
	char			ss[ITEMLEN];
	Line			*line = NULL;
	WINDOW		*vwin = common.windows[VERL];
	struct stat	crt;

	for (i = 0; i <  NUMDEMOD; i++)
		if (MASK_ISSET(USE, channel, i) && dem[i]->demod) {
			verbprintf(10, "Modul %2i, Zeile %i/%i", i, roll + 1, len_list);

			shift = NUMDEMOD * channel + i;

			if (chkdate) {
				/*	einmal pro Durchlauf:	*/
				int fd0, day;

				/* 1. Zeit schreiben	*/
				#define	VMAX	vwin->_maxy
				WINDOW *time_win = common.windows[LGND];
				wattrset(time_win, A_BOLD);
				memset(ss, 0, sizeof(ss));
				mvwaddstr(time_win, 0, 0, get_time(ss));
				wrefresh(time_win);
				memset(ss, 0, sizeof(ss));
				mvwaddstr(time_win, 0, 9, get_date(ss) + 2);
				wrefresh(time_win);
				day = ss[8] + ss[9];
				if (common.day && common.day != day) {
					/*	Ausgabe bei Tageswechsel	*/
					char outprint[LINELEN];
					sprintf(outprint, " Tageswechsel: %s", ss);
					update_view(outprint, 0, NOLOG, common.color[COL_PTT]);
				}
				common.day = day;
				chkdate = FALSE;

				/*	2. Tastatur lesen	*/
				common.key = getch();
				if (common.key != -1)
					verbprintf(9, "Fensterlnge: %i, Scroll: %i, Listenlnge: %i, Eingabe: %i, Eingabe0: %i", len_list, roll, VMAX, common.key, common.key0);
				switch (common.key) {

				case 'q':
					clear();
					refresh();
					endwin();
					exit(0);
					break;

				case 3:	/*	Zeile hoch	*/
					if (roll) roll--;
					break;

				case 2:	/*	Zeile runter	*/
					if (roll < END_LIST(len_list, VMAX))
						roll++;
					break;

				case 53:	/*	Tag vor	*/
					if (common.key0 == 91) {
						common.key0 = 0;
						if (roll > VMAX) {
							char outprint[LINELEN];
							int zeile = 0, change = 0;
							sprintf(outprint, " Tageswechsel:");
							line = common.line;
							while (line != NULL && zeile < roll) {
								if (!strncmp(line->string, outprint, 14)) change = zeile;
								zeile++;
								line = line->next;
							}
							roll = change;
						}
						else roll = 0;
					}
					break;

				case 54:	/*	Tag zurck	*/
					if (common.key0 == 91) {
						common.key0 = 0;
						if (len_list > VMAX) {
							int zeile = 0;
							char outprint[LINELEN];
							sprintf(outprint, " Tageswechsel: ");
							line = common.line;
							while (line != NULL && zeile < roll) {
								line = line->next;
								zeile++;
							}
							while (line != NULL && zeile < len_list && strncmp(line->string, outprint, 15)) {
								zeile++;
								line = line->next;
							}
							if (line != NULL && !strncmp(line->string, outprint, 15))
								roll = zeile;
							else
								roll = END_LIST(len_list, VMAX);
						}
						else roll = 0;
					}
					break;

				case 83:	/*	Page hoch	*/
					if (roll > VMAX)
						roll -= VMAX - 2;
					else roll = 0;
					break;

				case 82:	/*	Page down	*/
					if (len_list > VMAX) {
						if (roll < (len_list - 2 * VMAX))
							roll += VMAX - 2;
						else roll = END_LIST(len_list, VMAX);
					}
					break;

				case 6:	/*	Home	*/
				case 72:	/*	Home	*/
					roll = 0;
					break;

				case 104:/*	End	*/
				case 70:	/*	End	*/
					if (len_list > VMAX) roll = END_LIST(len_list, VMAX);
					break;

				case 91: /*	Memo	*/
					common.key0 = common.key;
					break;
				}
				
				if ((common.key >= 2 && common.key <= 3) 
				|| (common.key >= 82 && common.key <= 83)
				|| (common.key >= 6 )
				|| (common.key >= 104 )) {
					line = common.line;
					if (line != NULL && len_list > VMAX) {
						int zeile = roll;
						while (line->next != NULL && zeile) {
							zeile--;
							line = line->next;
						}
						while (line != NULL && zeile < VMAX) {
							wattrset(vwin, common.color[line->value]);
							mvwaddstr(vwin, zeile, 0, line->string);
							zeile++;
							line = line->next;
						}
						zeilen(roll + 1);
						wrefresh(vwin);
					}
				}

				/*	3. rc neu lesen	*/
				fd0 = stat(common.rcfile.name, &crt);
				if (windowResized() || (!fd0 && common.rcfile.create_time != crt.st_mtime)) {
					/*	jetzt wird alles gut ;-)	*/
					int	j, ichan, _shift;
					common.rcfile.create_time = crt.st_mtime;
					endwin();
					initall();
					getch(); /*	ntig fr Display	*/
					dump = 0;
					/* Vergleichen von alten und neuen Modulen;
					 *	ggf. Schlieen, Starten oder Neustarten	*/
				 	for (ichan = 0; ichan < CHANNELS; ichan++) {
						for (j = 0; j < NUMDEMOD; j++) {
							if (MASK_ISSET(USE, ichan, j)) {
								/* Schlieen */
								_shift = ichan * NUMDEMOD + j;
								dem[j]->kill(dem_st + _shift);
								if (MASK_ISSET(HELP, ichan, j)) {
									/*	Neustart	*/
									init_struct(ichan, j);
								}
							}
							else {
								if (MASK_ISSET(HELP, ichan, j)) {
									/*	Starten	*/
									init_struct(ichan, j);
								}
							}
						}
						dem_mask[USE][ichan] = dem_mask[HELP][ichan];
					}
					return;
				}
			}
			dem[i]->demod(dem_st + shift, buf, len);
		}
}	/* process_buffer

---------------------------------------------------------------------- */
#ifdef SUN_AUDIO

void input_sound(unsigned int	sample_rate, const char	*ifname) {
/***********************************************************************	
 *	SUN-Audio-Device abfragen
 ***********************************************************************/

audio_info_t audioinfo;
audio_info_t audioinfo2;
audio_device_t audiodev;
int fd;
short buffer[8192 * CHANNELS];
float fbuf[CHANNELS][16384];
unsigned int fbuf_cnt = 0;
int i;
short *sp;
	
	if ((fd = open(ifname ? ifname : "/dev/audio", O_RDONLY)) < 0) {
		perror("open");
		exit(10);
	}
	if (ioctl(fd, AUDIO_GETDEV, &audiodev) == -1) {
		perror("ioctl: AUDIO_GETDEV");
		exit(10);
	}
	AUDIO_INITINFO(&audioinfo);
	audioinfo.record.sample_rate = sample_rate;
	audioinfo.record.channels  = 2;
	audioinfo.record.precision = 16;
	audioinfo.record.encoding  = AUDIO_ENCODING_LINEAR;

	audioinfo.record.gain = 0x20;
	audioinfo.record.port = AUDIO_LINE_IN;
	audioinfo.monitor_gain = 0;

	if (ioctl(fd, AUDIO_SETINFO, &audioinfo) == -1) {
		perror("ioctl: AUDIO_SETINFO");
		exit(10);
	}
	if (ioctl(fd, I_FLUSH, FLUSHR) == -1) {
		perror("ioctl: I_FLUSH");
		exit(10);
	}
	if (ioctl(fd, AUDIO_GETINFO, &audioinfo2) == -1) {
		perror("ioctl: AUDIO_GETINFO");
		exit(10);
	}

/*	fprintf(stdout, "Audio device: name %s, ver %s, config %s, "
		"sampling rate %d\n", audiodev.name, audiodev.version,
		audiodev.config, audioinfo.record.sample_rate);
*/

	for (;;) {
		channel = 0;
		i = read(fd, sp = buffer, sizeof(buffer));
		if (i < 0 && errno != EAGAIN) {
			perror("read");
			exit(4);
		}
		if (!i) break;
		if (i > 0) {
			for (; i >= sizeof(buffer[0]); i -= sizeof(buffer[0]), sp++) {
				fbuf[0][fbuf_cnt] = (*sp++) * (1.0/32768.0);
				fbuf[1][fbuf_cnt++] = (*sp) * (1.0/32768.0);
			}
			if (i)
				fprintf(stderr, "warning: noninteger number of samples read\n");
			if (fbuf_cnt > overlap) {
				process_buffer(fbuf[0], fbuf_cnt - overlap);
				memmove(fbuf[0], fbuf[0] + fbuf_cnt - overlap, overlap * sizeof(fbuf[0][0]));
				fbuf_cnt = overlap;
				channel++;
				process_buffer(fbuf[1], fbuf_cnt - overlap);
				memmove(fbuf[1], fbuf[1] + fbuf_cnt - overlap, overlap * sizeof(fbuf[1][0]));
				fbuf_cnt = overlap;
			}
		}
	}
	close(fd);
}

#else /* SUN_AUDIO */
/* ------------------------------------------------------------------- */

void input_sound(unsigned int	sample_rate, const char *ifname) {
/***********************************************************************	
 *	Audio-Device abfragen
 ***********************************************************************/

int sndparam;
int fd;

/*	verdoppelt fr 2-Kanal	*/
union {
	short s[8192 * CHANNELS];
	unsigned char b[8192 * CHANNELS];
} b;

float fbuf[CHANNELS][16384];
unsigned int fbuf_cnt = 0;
int i;
short *sp;
unsigned char *bp;
int fmt = 0;

	if ((fd = open(ifname ? ifname : "/dev/dsp", O_RDONLY)) < 0) {
		perror("open");
		exit(10);
	}
	sndparam = AFMT_S16_LE; /* we want 16 bits/sample signed */
	/* little endian; works only on little endian systems! */
	if (ioctl(fd, SNDCTL_DSP_SETFMT, &sndparam) == -1) {
		perror("ioctl: SNDCTL_DSP_SETFMT");
		exit(10);
	}
	if (sndparam != AFMT_S16_LE) {
		fmt = 1;
		sndparam = AFMT_U8;
		if (ioctl(fd, SNDCTL_DSP_SETFMT, &sndparam) == -1) {
			perror("ioctl: SNDCTL_DSP_SETFMT");
			exit(10);
		}
		if (sndparam != AFMT_U8) {
			perror("ioctl: SNDCTL_DSP_SETFMT");
			exit(10);
		}
	}
	sndparam = 1;   /* 2 Kanle */
	if (ioctl(fd, SNDCTL_DSP_STEREO, &sndparam) == -1) {
		perror("ioctl: SNDCTL_DSP_STEREO");
		exit(10);
	}
	if (sndparam != 1) {
		/*	Monovariante?	*/
		fprintf(stderr, "soundif: Error, cannot set the channel "
			"number to 2\n");
		exit(10);
	}

	sndparam = sample_rate; 
	if (ioctl(fd, SNDCTL_DSP_SPEED, &sndparam) == -1) {
//		perror("ioctl: SNDCTL_DSP_SPEED");exit(10);
	}
	if ((10 * abs(sndparam - sample_rate)) > sample_rate) {
//		perror("ioctl: SNDCTL_DSP_SPEED");exit(10);
	}

	if (sndparam != sample_rate) {
		fprintf(stderr, "Warning: Sampling rate is %u, "
			"requested %u\n", sndparam, sample_rate);
	}

#if 0
	sndparam = 4;
	if (ioctl(fd, SOUND_PCM_SUBDIVIDE, &sndparam) == -1) {
		perror("ioctl: SOUND_PCM_SUBDIVIDE");
	}
	if (sndparam != 4) {
		perror("ioctl: SOUND_PCM_SUBDIVIDE");
	}
#endif

	for (;;) {
		channel = 0;
		if (fmt) {
			i = read(fd, bp = b.b, sizeof(b.b));
			if (i < 0 && errno != EAGAIN) {
				perror("read");
				exit(4);
			}
			if (!i) break;
			if (i > 0) {
				for (; i >= sizeof(b.b[0]); i -= sizeof(b.b[0] * 2), bp++) {
					fbuf[0][fbuf_cnt] = ((int)(*bp++) - 0x80) * (1.0 / 128.0);
					fbuf[1][fbuf_cnt++] = ((int)(*bp) - 0x80) * (1.0 / 128.0);
				}

				if (i)
					fprintf(stderr, "warning: noninteger number of samples read\n");
				if (fbuf_cnt > overlap) {
					process_buffer(fbuf[0], fbuf_cnt - overlap);
					memmove(fbuf[0], fbuf[0] + fbuf_cnt - overlap, overlap * sizeof(fbuf[0][0]));
					channel++;
					process_buffer(fbuf[1], fbuf_cnt - overlap);
					memmove(fbuf[1], fbuf[1] + fbuf_cnt - overlap, overlap * sizeof(fbuf[1][0]));
					fbuf_cnt = overlap;
				}
			}
		} 
		else {
#ifdef ZLOG
char	outprint[ITEMLEN];
#endif
			i = read(fd, sp = b.s, sizeof(b.s));
			if (i < 0 && errno != EAGAIN) {
				/*	Schliessen des Sounddevices und neustarten	*/
				close(fd);
				return;
			}
#ifdef ZLOG
update_file("newbuf", LOG);
#endif
			if (!i) break;
			if (i > 0) {
				for (; i >= sizeof(b.s[0]); i -= sizeof(b.s[0]) * 2, sp++) {
					fbuf[0][fbuf_cnt] = (*sp++) * (1.0 / 32768.0);
					fbuf[1][fbuf_cnt++] = (*sp) * (1.0 / 32768.0);
#ifdef ZLOG
memset(outprint, 0, sizeof(outprint));
sprintf(outprint, "fbuf[%d]\t\t[li]%8.5f\t\t[re]%8.5f", fbuf_cnt-1, fbuf[0][fbuf_cnt-1], fbuf[1][fbuf_cnt-1]);
update_file(outprint, LOG);
#endif
				}
				if (i)
					fprintf(stderr, "warning: noninteger number of samples read\n");
				if (fbuf_cnt > overlap) {
					process_buffer(fbuf[0], fbuf_cnt - overlap);
					memmove(fbuf[0], fbuf[0] + fbuf_cnt - overlap, overlap * sizeof(fbuf[0][0]));
					channel++;
					process_buffer(fbuf[1], fbuf_cnt - overlap);
					memmove(fbuf[1], fbuf[1] + fbuf_cnt - overlap, overlap * sizeof(fbuf[1][0]));
					fbuf_cnt = overlap;
				}
			}
		}
	}
	close(fd);
}
#endif /* SUN_AUDIO */

/* ------------------------------------------------------------------- */

void input_file (unsigned int	sample_rate,
								const char		*fname,
								const char		*type) {
/***********************************************************************	
 *	Einlesen einer Sounddatei
 ***********************************************************************/

#undef STEREO_FILE
	struct stat statbuf;
	int pipedes[2];
	int pid = 0, soxstat;
	int fd;
	int i;
#ifdef STEREO_FILE
	short buffer[8192 * CHANNELS];
	float fbuf[CHANNELS][16384];
#else
	short buffer[8192];
	float fbuf[16384];
#endif
	unsigned int fbuf_cnt = 0;
	short *sp;

	/*
	 * if the input type is not raw, sox is started to convert the
	 * samples to the requested format
	 */
	if (!type || !strcmp(type, "raw")) {
		if ((fd = open(fname, O_RDONLY)) < 0) {
			verbprintf(ALWAYS, "\0");
			perror("open");
			exit(10);
		}
	} 
	else {
		if (stat(fname, &statbuf)) {
			verbprintf(ALWAYS, "\0");
			perror("stat");
			exit(10);
		}
		if (pipe(pipedes)) {
			verbprintf(ALWAYS, "\0");
			perror("pipe");
			exit(10);
		}
		if (!(pid = fork())) {
			char srate[8];
			/*
			 * child starts here... first set up filedescriptors,
			 * then start sox...
			 */
			sprintf(srate, "%d", sample_rate);
			close(pipedes[0]); /* close reading pipe end */
			close(1); /* close standard output */
			if (dup2(pipedes[1], 1) < 0) {
				verbprintf(ALWAYS, "\0");
				perror("dup2");
			}
			close(pipedes[1]); /* close writing pipe end */
			execlp("sox", "sox", 
			       "-t", type, fname,
			       "-t", "raw", "-s", "-w", "-r", srate, "-",
			       NULL);
			verbprintf(ALWAYS, "\0");
			perror("execlp");
			exit(10);
		}
		if (pid < 0) {
			verbprintf(ALWAYS, "\0");
			perror("fork");
			exit(10);
		}
		close(pipedes[1]); /* close writing pipe end */
		fd = pipedes[0];
	}
	/*
	 * demodulate
	 */
	for (;;) {
		channel = 0;
		i = read(fd, sp = buffer, sizeof(buffer));
		if (i < 0 && errno != EAGAIN) {
			verbprintf(ALWAYS, "\0");
			perror("read");
			exit(4);
		}
		if (!i)
			break;
		if (i > 0) {
#ifdef STEREO_FILE
				for (; i >= sizeof(buffer[0]); i -= sizeof(buffer[0]) * 2, sp++) {
					fbuf[0][fbuf_cnt] = (*sp++) * (1.0 / 32768.0);
					fbuf[1][fbuf_cnt++] = (*sp) * (1.0 / 32768.0);
				}
				if (i)
					fprintf(stderr, "warning: noninteger number of samples read\n");
				if (fbuf_cnt > overlap) {
					process_buffer(fbuf[0], fbuf_cnt - overlap);
					memmove(fbuf[0], fbuf[0] + fbuf_cnt - overlap, overlap * sizeof(fbuf[0][0]));
					channel++;
					process_buffer(fbuf[1], fbuf_cnt - overlap);
					memmove(fbuf[1], fbuf[1] + fbuf_cnt - overlap, overlap * sizeof(fbuf[1][0]));
					fbuf_cnt = overlap;
				}
#else
			for (; i >= sizeof(buffer[0]); i -= sizeof(buffer[0]), sp++)
				fbuf[fbuf_cnt++] = (*sp) * (1.0 / 32768.0);
			if (i)
				fprintf(stderr, "warning: noninteger number of samples read\n");
			if (fbuf_cnt > overlap) {
				process_buffer(fbuf, fbuf_cnt - overlap);
				memmove(fbuf, fbuf + fbuf_cnt - overlap, overlap * sizeof(fbuf[0]));
				fbuf_cnt = overlap;
			}
#endif
		}
	}
	close(fd);
	waitpid(pid, &soxstat, 0);
}	/* input_file

---------------------------------------------------------------------- */

int main (int argc, char *argv[]) {
/***********************************************************************	
 *	Parameterbergabe und Initialisierung der Datenstruktur und Module
 ***********************************************************************/

	char				*input_type = "hw";
	char				c, **itype;
	int				i, j;

	dump       = 0;
	first_init = 1;
	common.day = 0;
	laypage = 0;
	maxlaypage = 0;

	strcpy(common.rcfile.name, getenv("HOME"));
	strcat(common.rcfile.name, "/.monrc");

	/*	alle Module, die als Parameter bergeben wurden, werden
	 *	in dem_mask[USE] eingetragen.
	 *	Zunchst mask_first[USE] fr alle Kanle auf 1 setzen;
	 * wird erst 0, wenn irgendein Modul angegeben wird	*/
 	for (channel = 0; channel < CHANNELS; channel++)
		mask_first[USE][channel] = 1;

	while ((c = getopt(argc, argv, "t:f:ashn?")) != EOF) {

		switch (c) {

			case 'a':	/*	Modul hinzufgen	*/

				c = getopt(argc, argv, "L:R:M:?");

				if (c == -1 || c == '?') {
					errflg++;
					break;
				}

				for (i = 0; i < NUMDEMOD; i++)
					if (!strcasecmp(optarg, dem[i]->name)) {

						switch (c) {

						case 'L':
							channel = 0;
							break;

						case 'R':
							channel = 1;
							break;

						case 'M':
							channel = 2;
							break;

						default:
							verbprintf(ALWAYS, "monitor: Kanal '-L' oder '-R' angeben.");
							exit(1);	
						}

						if (mask_first[USE][channel]) {
							mask_first[USE][channel] = 0;
							dem_mask[USE][channel] = 0;
							/*	hier wird gleich noch der andere Kanal gelscht	*/
							j = (channel + 1) % 2;
							if (mask_first[USE][j]) {
								mask_first[USE][j] = 0;
								dem_mask[USE][j] = 0;
							}
						}
						MASK_SET(USE, channel, i);
						break;
					}

				if (i >= NUMDEMOD) {
//					endwin();
					fprintf(stderr, "invalid mode \"%s\"\n", optarg);
					errflg++;
				}
				break;

			case 's':	/*	Modul sperren	*/

				c = getopt(argc, argv, "L:R:M:?");

				if (c == -1 || c == '?') {
					errflg++;
					break;
				}

				for (i = 0; i < NUMDEMOD; i++) {
					if (i >= 0 && !strcasecmp(optarg, dem[i]->name)) {

						switch (c) {

						case 'L':
							channel = 0;
							break;

						case 'R':
							channel = 1;
							break;

						case 'M':
							channel = 2;
							break;

						default:
							printf("\nmonitor: Kanal '-L' oder '-R' angeben.\n");
							exit(1);	
						}

						if (i > -1) {
							if (mask_first[channel])
								memset(dem_mask[USE] + channel, 0xff, sizeof(dem_mask[USE][channel]));
							MASK_RESET(USE, channel, i);
						}
						mask_first[USE][channel] = 0;
						break;
					}
				}

				if (i >= NUMDEMOD) {
					fprintf(stderr, "invalid mode \"%s\"\n", optarg);
					errflg++;
				}
				break;

			case 'f':	/*	rc-file	*/
				strcpy(common.rcfile.name, getenv("HOME"));
				strcat(common.rcfile.name, "/");
				strcat(common.rcfile.name, optarg);
				break;

			case 't':	/*	sound file	*/
				for (itype = (char **)allowed_types; *itype; itype++) 
					if (!strcmp(*itype, optarg)) {
						input_type = *itype;
						goto intypefound;
					}
				fprintf(stderr, "Falscher Filetype \"%s\"\n"
					"erlaubte Typen: ", optarg);
				for (itype = (char **)allowed_types; *itype; itype++) 
					fprintf(stderr, "%s ", *itype);
				fprintf(stderr, "\n");
				errflg++;
				intypefound:
				break;

			default:
				errflg++;
				break;

		}	/*	switch	*/
	}	/*	while	*/

	/*	Jetzt werden u.a. die Module aus der .monrc geholt und
	 *	in dem_mask[HELP] gelegt; ebenso wird mask_first[HELP] gesetzt	*/
	initall();

	/*	Nun werden die aktivierten Module aus den Parametern [USE] und
	 *	aus der .monrc [HELP] ODER-verknpft */
	for (channel = 0; channel < CHANNELS; channel++) {
		dem_mask[USE][channel] |= dem_mask[HELP][channel];

		/*	Die mask_first-Eintrge werden UND-verknpft,
		 *	d.h. es bleibt nur eine 1 stehen,
		 *	falls mask_first nicht zurckgesetzt wurde,
		 *	also nirgendwo ein Module angegeben wurde	*/
		mask_first[USE][channel] &= mask_first[HELP][channel];

		/*	Falls also keine Module angegeben wurden,
		 *	werden in dem_mask[USE] alle Module auf
		 *	allen Kanlen auf 1 gesetzt	*/
		if (mask_first[USE][channel])
			memset(dem_mask[USE] + channel, 0xff, sizeof(dem_mask[USE][channel]));
	}

	for (channel = 0; channel < CHANNELS; channel++)
		for (i = 0; i < NUMDEMOD; i++)
			if (MASK_ISSET(USE, channel, i))
				init_struct(channel, i);

	if (!strcmp(input_type, "hw")) {

		/*	Endlosschleife: Daten lesen und auswerten	*/
		while(1) {
			if ((argc - optind) >= 1)
				input_sound(FREQ_SAMP, argv[optind]);
			else 
				input_sound(FREQ_SAMP, NULL);
		}
		exit(0);
	}
	if ((argc - optind) < 1) {
		(void)fprintf(stderr, "no source files specified\n");
		exit(4);
	}
	for (i = optind; i < argc; i++)
		input_file(FREQ_SAMP, argv[i], input_type);

	getchar();
//	initscr();	/* Calling refresh() will cause a segfault if we do it without initializing the screen. */
	clear();
	refresh();
	endwin();
	exit(0);
}
