;*********************************************************
;	PIC BASED POCSAG ENCODER apr-1998
;	COPYRIGHT Henry Carl Ott
;*********************************************************
	list p=16c84,r=dec
	errorlevel 1	
	include p16c84.inc
	__FUSES _CP_OFF&_PWRTE_ON&_WDT_OFF&_XT_OSC
;*********************************************************


;----- internal macros------
_send	macro   data			; send a single byte via rs-232 routine
	movlw	data
	call	rs_tx
	endm

nop_2	macro				; delay two cycles in one instruction	
	goto	$+1
	endm

bank0	macro
	bcf STATUS,RP0
	endm

bank1	macro
	bsf STATUS,RP0
	endm

_movff	macro src,dest			; move file reg to file reg
	movfw src
	movwf dest
	endm	

cje     MACRO   file,con,jump_to        ;compare, jump if equal
        movlw   con
        subwf   file,w
        skpnz
        goto    jump_to
        ENDM

;************************** registers *******************************

		cblock	0x0c	; start of general purpose registers
		temp_w		; store w for isr
		temp_status 	; store status for isr
		
		r0,r1,r2,r3	; scratch registers	
	
		bit_cnt		; all purpose bit counter	
		txreg	 	; rs232 data to transmit
		rxreg		; rs232 receive data

				; _data - _data3 = full encoded 32 bits to ship
		_data	 	; first byte of code word msb (keep registers sequential)
		_data1		; second byte of code word
		_data2		; the third 	
		_data3		; and forth lsb (mostly ecc)

		cap0		; cap code msb  (two bits)
		cap1		;               (8 bits)  
		cap2		; cap code lsb + function bits (20 bits total )
		cap_frame	; lsb frame bits of cap address (three bits)
	
		pocbuff		; tx buffer for isr tx routine
		pocbuff1
		pocbuff2
		pocbuff3

		poc_cnt		; bit counter for pocsag xmit routine
		frame		; actual tx frame counter
		baud_cnt	; divisor for isr baud generator
		profile		; which stored pager setup to send
		tic		; tic ++ every time through isr
		flags		; flag bits
		
		period_cnt	; debuging
		endc

 
;------ flag bit definitions
baud_0		equ	0x00	; pocsag baud rate 
baud_1		equ	0x01	; 00 = 512    01 = 1200  10 or 11 = 2400
invert		equ	0x02	; inverted or true data
comp_comm	equ	0x03	; if we are connected to computer / interactive mode

div0		equ	0x05	; used to inc div1 and 2 
div1		equ	0x06	; divide by 4
div2		equ	0x07	; for 512 baud rate generation		
;--------------ports--------------------------
;   port a
led_1		equ	0x00	; leds
led_2		equ	0x01	; leds
ptt		equ	0x02	; push to talk line (hard code correct polarity)
poc_data	equ	0x03	; output data

;   port b
switch1		equ	0x00	; push button
rs232_out	equ	0x01
rs232_inp	equ	0x02
switch2		equ	0x03	;
switch3		equ	0x04	;
;----------------misc--------------------------
cr		equ	0x0D
lf		equ	0x0A	
baud_div	equ	.96	; baud rate divisor for rs-232 routines and isr timer tic

frame_sync	equ	0x7CD215D8
idle 		equ	0x7A89C197

;-------------eeprom locations ------------------------------
valid_ee	equ	0x00	; profile check nibble 0x55 = programmed profile
cap_ee		equ	0x01	; eeprom locations / decimal cap code
func_ee		equ	0x08	; "	           / function bits
baud_ee		equ	0x09    ; "		   / baud rate
invert_ee	equ	0x0a	; "		   / invert data?		
msgflag_ee	equ	0x0b	; "                / does profile contain msg data?
msg1_ee		equ	0x0c	; holds profile msg numeric data (pre-encoded)	 
msg2_ee		equ	0x0d	; "
msg3_ee		equ	0x0e	; " 	
msg4_ee		equ	0x0f	; "
;**********************************************
		org 0
		goto 	init
		org 4
;************   timer interrupt  **************
		movwf 	temp_w		;save w & status
		swapf 	STATUS,w 	
		bcf 	STATUS,RP0
		movwf 	temp_status

;---------------------------------------
; TMR0	3.686400 mhz clock	
; 1/9600 = .000104167 ---  
; 1/38400 = .000026041667 
; 1/19200 = .000052083333

		movlw 	(.256 - baud_div) + .10	; the 10 = interupt and isr overhead	
		movwf 	TMR0			; 
	
int_timer	bcf 	INTCON,T0IF		; clear interupt flag	
		incf 	tic,f			;+ 1 every time through isr
		incf	baud_cnt,f		; pocsag baud divider
				

		btfss	PORTB,rs232_inp		; test here for rs-232 activity
		bsf	flags,comp_comm		; to set flag for interactive computer mode
		
		
set_freq	btfsc	flags,baud_0		; test baud rate bits
		goto	set_1200	 		;

		btfsc 	flags,baud_1		;
		goto	set_2400	
					
									
set_512		movlw	0xc0			; no baud bits? then its 512	
		andwf	flags,w
		movlw	.19
		skpnz				;(9600*4) / ((3*19)+18)) = 512
		movlw	.18

		xorwf	baud_cnt,w
		skpz
		goto	int_end			; no match keep counting

		movlw	0x40
		addwf	flags,f			; divide by 4
		goto	period


set_1200	movlw 	.8 		;1200 = 9600 / 8
		xorwf 	baud_cnt,w	; get count
		skpz
		goto 	int_end		
		goto 	period		; match


set_2400	movlw	.4		; 2400 = 9600 / 4
		xorwf	baud_cnt,w
		skpz			
		goto	int_end		; no match
		

period		incf	period_cnt		; debugging
		btfss	period_cnt,0		; toggle output at send baud rate
		bcf	PORTA,led_1		; 
		btfsc	period_cnt,0		;		
		bsf	PORTA,led_1		; end debugging


		clrf	baud_cnt		; reset baud rate divisor
		movf	poc_cnt,f		; test for zero
		skpnz				; 
		goto	int_end			; nothing to send

		rlf	pocbuff3		; rotate msb of buffer into carry
		rlf	pocbuff2
		rlf	pocbuff1
		rlf	pocbuff			; msb

		btfsc	flags,invert		; test to see if inverted data flag is set
		goto	invert_data		; if flag bit is set, we invert the data

		skpc				; test and set output pin
		bcf	PORTA,poc_data		; true data
		skpnc
		bsf	PORTA,poc_data		; double test carry for no jitter	

		goto	sb_end


invert_data	skpc				; test and set output pin
		bsf	PORTA,poc_data		; negative data	
		skpnc
		bcf	PORTA,poc_data		; double test carry for no jitter


sb_end		decf	poc_cnt
		

;---------------------------------------
; end of interrupt
;---------------------------------------
int_end		swapf 	temp_status,w		; restore w and status
		movwf 	STATUS
		swapf 	temp_w
		swapf 	temp_w,w
		retfie				; re-enable ints and return

;***************************************
init
;----------------clear ram from 0x0C to 0x2F --------------------
		movlw   0x0c            ;get first ram location
		movwf   FSR             ;load in indirect address
CR1		clrf    INDF            ;do indirect ram clr
		incf    FSR             ;inc FSR
		btfss   FSR,5           ;if bit 5 set then skip
		goto    CR1             ;else clr next
		btfss   FSR,4           ;if bit 4 set then done
		goto    CR1             ;else clr next
;----------------- program specific inits ----------------------- 
		clrf	PORTA		; turn off leds
		bsf	PORTB,rs232_out	; set up port b (rs232 line goes hi)
			
		bank1			; set up ports
		movlw 	b'00000000'		
		movwf	TRISA		; port a all output 
		movlw 	b'00011101' 
		movwf	TRISB		; port b all outs except for rs232-input and sw_inp 

		movlw 	b'00001000'	; pullups enabled, prescaler goes to wdt (not used) 
		movwf	OPTION_REG
		bank0			; back to bank 0 for normal operations
	
startint	call	banner		; display id string

		movlw 	(256-baud_div) 	; tmr0
		movwf 	TMR0		; set up timer for basic tic element
 		movlw 	b'10100000'
		movwf	INTCON		; enable global, and  timer overflow interupts
		

main		btfsc	flags,comp_comm	; test to see if we the computer wants to talk
		goto	comm		; goto interactive computer routines
		
;-------------------------------------------------------
; this is for testing purposes nothing fancy / no debouncing.
; in final application  you just select the profile number and call do_page.
;-------------------------------------------------------
; test for push button closure, send associated pager profile
; 
		clrf	profile		 
		btfss	PORTB,switch1
		call	do_page

		incf	profile,f
		btfss	PORTB,switch2
		call	do_page

		incf	profile,f
		btfss	PORTB,switch3
		call	do_page

main_end	goto	main		; loop it
				

;=======================================
; Print Info
; get the copyright data and ship it
;---------------------------------------
banner		movlw 	copy_r-0x300	; 
		movwf 	r0		; pointer to message 

next_data	movfw 	r0
		call 	get_data	; get the data
		iorlw	0x00		; check for end
		skpnz
		goto	end_of_data
		incf 	r0		; point to next
		call 	rs_tx		; char send routine	
		goto 	next_data	; keep looping	
end_of_data	return

;---------------------------------------------------------------
;interactive computer mode, we get to talk to computer with interupts OFF 
;---------------------------------------------------------------
comm		bcf	INTCON,GIE		;turn off interupts
comm_top	call	crlf	
		_send	'.'			; standard prompt

comm_2		call	get_w_echo		
						; echo to console, case counts!
		cje	rxreg,'x',comm_end	; exit
		cje	rxreg,'p',progdata	; program capcode data
		cje	rxreg,'d',dump_ee	; dump first 16 eeprom locations / for debugging
 	
		goto	comm_top
	
comm_end	bcf	flags,comp_comm		; clear the comm flag
		goto	startint		; and restart the pic	

;---------------get a complete  pager profile from the user --------
progdata	call	crlf		; make pretty

		_send	'P'		; get pager profile number
		_send	'R'
		_send	'O'
		_send	'F'
		_send	'I'
		_send	'L'
		_send	'E'
		_send	'='

		call	get_asc
		movwf	EEADR
		swapf	EEADR,f		; multiply by 0x10

		clrf	EEDATA
		call	wr_ee		; clear out profile check byte


		_send	' '
		_send	'C'
		_send	'A'
		_send	'P'
		_send	'='

		movlw	.07		;seven digits to retreive
		movwf	r3
pd_a		call	get_ee_nibble	; cap msd		
		decfsz	r3
		goto	pd_a		; keep going

		_send	' '		; prompt
		_send	'F'
		_send	'U'
		_send	'N'
		_send	'C'
		_send	'='

		call	get_ee_nibble	

		_send	' '		; get baud rate
		_send	'B'
		_send	'A'
		_send	'U'
		_send	'D'
		_send	'='

		call	get_ee_nibble


		_send	' '		; get invert status
		_send	'I'
		_send	'N'
		_send	'V'
		_send	'='

		call	get_ee_nibble

	
		_send	' '		; prompt if message contains data
		_send	'M'
		_send	'S'
		_send	'G'
		_send	'?'
		_send	' '
		_send	'Y'
		_send	'/'
		_send	'N'
		_send	' '

		call	get_w_echo	; get a char
		bcf	rxreg,5		; force to upper case
		movlw	'Y'		; anything but 'Y' or 'y' aborts 
		xorwf	rxreg,w
		skpz
		goto	no_msg		; no msg adata

		movlw	0x01		; mark message flag
		movwf	EEDATA
		call	wr_ee_w	


get_msg		call	crlf		; prompt
		_send	'Q'
		_send	'='
		_send	'Q'
		_send	'U'
		_send	'I'
		_send	'T'
		_send	' '

		_send	'N'
		_send	'U'
	 	_send	'M'
		_send	'-'
		_send	'>'	


gm_1		call	get_msg_data	; get first nibble
		skpnc
		goto	get_msg		; error restart
		movwf	_data1

		call	get_msg_data	; get second nibble
		skpnc
		goto	get_msg
		movwf	_data2
		swapf	_data2,f	; into position
		
		call	get_msg_data	; get third nibble
		skpnc
		goto	get_msg
		iorwf	_data2,f	; stuff it

		call	get_msg_data	; get fourth nibble
		skpnc
		goto	get_msg
		movwf	_data3
		swapf	_data3,f	; into position
		
		call	get_msg_data	; get fifth nibble
		skpnc
		goto	get_msg
		iorwf	_data3,f	; stuff it

	
		bsf	_data1,4	; set bit to indicate data code word
		call	fix_data	; calc ecc and parity	
		
		movfw	_data		; fixed up data comes back in _data - _data3
		call	wr_ee_w		; stuff it into 4 eeprom locations for later transmisssion

		movfw	_data1
		call	wr_ee_w

		movfw	_data2
		call	wr_ee_w

		movfw	_data3
		call	wr_ee_w
				
		decf	EEADR		; undo last incf to stay in same eeprom page 
					;  
		goto	pd_done

no_msg		clrf	EEDATA
nm_1		call	wr_ee


pd_done		movlw	0xf0		; mark the profile as written
		andwf	EEADR,f		; 
		movlw	0x55
		call	wr_ee_w

		goto	comm_top	; and back up to the top


;------ dump first 16 eeprom locations for debugging purposes
dump_ee		movlw	0x10	;16 locations
		movwf	r2	; save (send_asc uses r1)
		clrf	EEADR	; start at bottom
		call	crlf

dump_a		movfw	EEADR
		call	send_asc
		_send	'-'
		call	rd_ee
		call	send_asc
		_send	' '
		_send	' '
		movfw	EEADR
		xorlw	0x08	; wrap around?
		skpnz
		call	crlf
		decfsz	r2	; any more?
		goto	dump_a
		goto	comm_top		 				
	
;---------get a nibble convert it and write to eeadr 
get_ee_nibble	call	get_asc
		call	wr_ee_w
		return
		
;-----get two ascii chars and write to eeprom, then inc eeadr
get_ee_word	call	get_asc		;
		movwf	EEDATA	
		swapf	EEDATA,f
		call	get_asc
		iorwf	EEDATA,f
		call	wr_ee
		return

;----- get a char with echo --------
get_w_echo	call	rs_rx
		skpnc
		goto	rs_time_out	; timout error -  restart
		call	rs_tx
		movfw	rxreg		;	restore w
		return

;---- get single ascii char with echo and convert to hex returned in w------
get_asc		call	rs_rx
		skpnc
		goto	rs_time_out
		call	rs_tx
		movfw	rxreg		; restore w
		call	asc_hex		; convert
		skpnc
		goto	comm_top	;error start over
		return

;-----get numeric data for message ---------
; also converts space and hyphen characters
; other special chars have to be entered as a hex nibble 
get_msg_data	call	rs_rx
		skpnc
		goto	rs_time_out	
		call	rs_tx		; echo

		movlw	'Q'		; test for quit
		xorwf	rxreg,w
		skpnz
		goto	comm_top	; esc, goto top
		
		movlw	' '		; test for space
		xorwf	rxreg,w	
		skpz
		goto	gmd_1
		movlw	0x0c
		goto	bit_swap

gmd_1		movlw	'-'		; test for hyphen
		xorwf	rxreg,w	
		skpz
		goto	gmd_2
		movlw	0x0d
		goto	bit_swap


gmd_2		movfw	rxreg		; restore w
		call	asc_hex	
		skpnc
		return			; error in conversion


bit_swap	movwf	rxreg		
		clrf	r3

		btfsc	rxreg,0		;reverse bit order (dumb)
		bsf	r3,3
		btfsc	rxreg,1
		bsf	r3,2
		btfsc	rxreg,2
		bsf	r3,1
		btfsc	rxreg,3
		bsf	r3,0
		movfw	r3		; get results
		clrc
		return			; return okay

;-------- send timeout message and then restart device -------
rs_time_out
		call	crlf		; time out error message
		_send	0x07		; ding ding
		_send	0x07
		_send	"T"
		_send	"O"
		call	crlf
		goto	comm_end	; restart device

		
;-----------------  send cr then line feed -------------------
crlf		_send	cr
		_send	lf
 		return


;------- actually send a page ---------------------
;------- set profile register before calling ------
do_page		
		swapf	profile,w	; get profile	
		call	rd_ee_w
		xorlw	0x55		; test?
		skpz
		return			; nonvalid profile entry, we just return

		swapf	profile,w	; get profile, and multiply by .16
		addlw	baud_ee		; get baud pointer into eeprom

		call	rd_ee_w		; reads the data pointed to by w
		movwf	r0		; save baud data

		call	rd_ee		; get invert data status
		movf	EEDATA,f	; test for zero	
		skpz 
		bsf	r0,invert	; set invert flag		
					
		movfw	r0
		movwf	flags		; and actually set the flag bits
		
		bsf	PORTA,ptt	; set ptt bit
		movlw	.254
		call	tic_delay	; let xmitter get stable, also baud syncs up
		
		call	bcd_bin		; do binary conversions		
					
		_movff	cap0,_data1	; move cap code to working regs
		_movff	cap1,_data2	; for encoding
		_movff	cap2,_data3		

		bcf	_data1,4	; clear msb to indicate an address field	
		call	fix_data	; generate ecc, add in parity, 	

;--------------------------- send the preamble ---------------------
		movlw	.18		; 18 * 32 = 576
		movwf	r0
sp_a		call	stuff_aa	; preamble 
		decfsz	r0		
		goto	sp_a
		
send_sync	clrf	frame		; clear frame counter
		call	stuff_sync	; send a sync word
	
send_frame	clrc
		rlf	cap_frame,w	; get cap_frame shifted
		xorwf	frame,w		; match?
		skpz
		goto	send_idle
		call	stuff_data	; match! send cap code	

		swapf	profile,w
		addlw	msgflag_ee
		call	rd_ee_w		; get msg status flag

		movf	EEDATA,f	; test for zero
		skpnz
		goto	next_frame	; no data keep going

		incf	frame,f		; 
		call	stuff_numbers
		goto	next_frame

send_idle	call	stuff_idle	

next_frame	incf	frame,f		; inc frame counter	
		btfss	frame,4		; roll over at .16	
		goto	send_frame	; keep going


do_page_end	movf	poc_cnt,f
		skpz
		goto	do_page_end	; wait until isr is done transmitting last word	
					
		bcf	PORTA,poc_data	; clear data line (not necessary)	
		btfsc	flags,invert	; 
		bsf	PORTA,poc_data

		bcf	PORTA,ptt	; drop ptt line			
		return

; indidual routines to stuff specific data into transmit buffer
; very sloppy, in the future re-code to a more generic routine that can fed table pointers

stuff_aa	call	stuff_wait
		movlw	0xAA
		movwf	pocbuff
		movwf	pocbuff1
		movwf	pocbuff2
		movwf	pocbuff3
		goto	stuff_end

stuff_sync	call	stuff_wait
		movlw	0x7c
		movwf	pocbuff
		movlw	0xd2
		movwf	pocbuff1
		movlw	0x15
		movwf	pocbuff2
		movlw	0xd8
		movwf	pocbuff3
		goto	stuff_end

stuff_idle	call	stuff_wait
		movlw	0x7a
		movwf	pocbuff
		movlw	0x89
		movwf	pocbuff1
		movlw	0xc1
		movwf	pocbuff2
		movlw	0x97
		movwf	pocbuff3
		goto	stuff_end

stuff_numbers	call	stuff_wait

		swapf	profile,w		; get profile
		addlw	msg1_ee			; add in the index to data

		call	rd_ee_w			; get first byte of msg
		movwf	pocbuff
		call	rd_ee
		movwf	pocbuff1
		call	rd_ee
		movwf	pocbuff2
		call	rd_ee
		movwf	pocbuff3
		goto	stuff_end


stuff_data	call	stuff_wait
		_movff	_data,pocbuff
		_movff	_data1,pocbuff1
		_movff	_data2,pocbuff2
		_movff	_data3,pocbuff3

stuff_end	bsf	poc_cnt,5			;set transmit flag 		
		return

;----------wait for isr transmit buffer to clear
stuff_wait	movf	poc_cnt,f
		skpz
		goto	stuff_wait
		return

;--- convert 7 digit capcode pointed to by profile into binary  -------
;--- save in cap - cap2 seperate out frame data, and store in cap_frame -------
;--- and while we are at it, mask in function bits from eeprom ----- 
;--- There has to be a more efficient way to do this, but I had code space to burn.

bcd_bin		clrf	_data	; clear out working registers
		clrf	_data1	
		clrf 	_data2
		
		swapf	profile,w 	; get page set / which profile?	
		addlw	cap_ee		; pointer to cap code bcd data
		movwf	EEADR		; point to lsd			

		movlw	0x0F
		movwf	r2
		movlw	0x42
		movwf	r1
		movlw	0x40	
		movwf	r0
		call	ee_mult		;(millions)

		movlw	0x01
		movwf	r2
		movlw	0x86
		movwf	r1
		movlw	0xA0	
		movwf	r0
		call	ee_mult		;(hund _ thousands)
	
		clrf	r2
		movlw	0x27
		movwf	r1
		movlw	0x10	
		movwf	r0
		call	ee_mult		;(ten _ thousands)

		movlw	0x03
		movwf	r1
		movlw	0xe8	
		movwf	r0
		call	ee_mult		;(thousands)

		clrf	r1
		movlw	0x64	
		movwf	r0
		call	ee_mult		;(hundreds)

		movlw	0x0a
		movwf	r0
		call	ee_mult		;(tens)


		call	rd_ee		; get the ones
		movwf	r0
		call	add24		; just a little faster this way

;----------- all done with the math --------------------------
;----------- seperate out data and stuff into data registers ----	
		movfw	_data2		
		andlw	b'00000111'	; mask off the top 5 bits
		movwf	cap_frame	; save frame data (3 bits)	

		call	rr_data		; shift to make room for function bits
		
		movlw	b'11111100'
		andwf	_data2,f	; mask off low order bits		

		swapf	profile,w	; get current profile
		addlw	func_ee		; add in pointer to function bits
		
		call	rd_ee_w		; get function
		andlw	b'00000011'	; just in case, mask of top  bits

		iorwf	_data2,f	; and mask  it in

		_movff	_data,cap0	; and move converted data
		_movff	_data1,cap1	; to correct file registers
		_movff	_data2,cap2
	
		return			; all done

;---------- get cap code digit from eeprom and do repeatative adds -----------------
;
ee_mult		call	rd_ee		; get mult
		movf	EEDATA,f	; testing for zero
		skpnz
		return			; zero just return
mult_a		call	add24		; do the math
		decfsz	EEDATA,f
		goto	mult_a
		return

;------- the actual 24 bit addition --------------------
; data in r0,r1,r2 is added to _data,_data1,_data2
; we ignore any  overflow in msb
add24		movfw 	r1		; get middle byte	
		addwf	_data1,f	; add to middle	

		skpnc
		incf	_data		; incd top byte if carry

		movfw	r2		; bet top byte src
		addwf	_data,f		; add to top byte dest

		movfw 	r0		; get bottom byte src
		addwf	_data2,f	; add to bottm dest	

		skpc			; test for _data1 to _data carry
		return	
		incf	_data1		; 
		skpnz
		incf	_data
		return			

; ---------------- fix up the data ------------------
; takes  21 bits of _data adds ecc and parity. 
; data starts in  _data1, _data2, and data3 
; ends up in _data,_data1,_data2, _data3

fix_data       	call	rl_data		; 
		call	rl_data	
		call	rl_data		; shift data into initial position

		movfw	_data1		; 
		movwf	r1		; save data
		movfw	_data2
		movwf	r2		; save data		
		movfw	_data3
		movwf	r3		; save data

		movlw	.21		; total number of bits to test	
		movwf	bit_cnt
fix_a		btfss	_data1,7	; test msb
		goto	fix_b		; no high bit, skip xor 
		movlw	0xED		; if the msb =1 the xor in 0xed20	
		xorwf	_data1,f	
		movlw	0x20		; 
		xorwf	_data2,f

fix_b		call	rl_data		; next bit	
		decfsz	bit_cnt		; any more bits?	
		goto	fix_a		; nope, keep going
					
;-------------- we are done with the xoring now put the ecc data ---				
;-------------- into the correct bit positions ----------------
		movlw	.13		; 13 bits to the right, not fast but simple
		movwf	bit_cnt
fix_c		call	rr_data		; rotate ecc data into correct position
		decfsz	bit_cnt
		goto	fix_c
	
		movfw	r1		; restore the saved orig data to correct position
		movwf	_data
		movfw	r2
		movwf	_data1

		movlw	b'00000111'	; mask off top 5 bits
		andwf	_data2		; make room for data

		movfw	r3
		iorwf	_data2,f	; and inclusive or in the final data word

;---------------- new parity routine (no data shifting)-----
;---------------- calc even parity for lsb --------------
parity		bcf	_data3,0	; might not be necessary	
		movfw	_data
		xorwf	_data1,w
		xorwf	_data2,w
		xorwf	_data3,w	; clear out redundent bits 
 
		movwf	r0		;save it

 		swapf 	r0,w     	;swap and xor (now only 4 bits to test!)
 		xorwf 	r0,w     
		
		movwf	r0		; data is in both r0 and w 

		rrf     r0,f    
		rrf     r0,f    
 		xorwf   r0,f    	; 
 		rrf     r0,w    
                xorwf   r0,f    	;parity left in bottom four bits 0000=even 1111=odd

		btfsc	r0,0
		bsf	_data3,0
parity_done	return


;--------- the old parity routine.  shorter but many more cycles -------
;		movlw	.31		; 31 bits to count					
;		movwf	bit_cnt		; bit counter
;		clrf	r0		; ones counter init
;parity		call	rr_w_c		; rotate with the carry bit
;		btfsc	_data3,0	; test bit, if a 1	
;		incf	r0		; bump ones counter
;		decfsz	bit_cnt		; keep going?
;		goto	parity
;
;		call	rr_w_c		; two final shifts to restore data
;		call	rr_w_c
;
;		btfsc	r0,0		; if the count was odd,
;		bsf	_data3,0	; set bit to make even	
;		return

;------------ rotates data word (all 32 bits) left
rl_data		clrc			; clear carry as we go
rl_w_c		rlf	_data3		; keep carry bit
		rlf	_data2
		rlf	_data1
		rlf	_data
		return
;------------ ditto but to the right
rr_data		clrc			 
rr_w_c		rrf	_data		; keep carry bit
		rrf	_data1
		rrf	_data2
		rrf	_data3
		return

;-----------------------------------------------------------
; ***** Receive serial data with time out ********* 
; gets a byte from rs-232 input, if no data in time out period,
; c = 0 okay data, c = 1 timout	
;-----------------------------------------------------------
rs_rx		movlw	0x50			; time out value
		movwf	r2			; hi order time out
		clrf	r1			; more time out
		clrf	r0			; time out innermost loop		

rx_a		decfsz	r1,f			; check for time out
		goto	rx_b			; keep looping
		decfsz	r2,f
		goto	rx_b	
		setc				; timout!!
		return

rx_b		decf	r0
		skpnz
		goto	rx_a			; time out
  		btfsc   PORTB,rs232_inp 	; test input pin
        	goto    rx_b         		

		movlw	((baud_div / 2) - 8)/4	
        	call    delay           	; delay a half bit to center of data window
        	btfsc   PORTB,rs232_inp    	; test input pin again should be in middle of start bit (and still high)
        	goto    rx_b            	; nope start over
        	movlw   .8              	; we want to get eight bits
        	movwf   bit_cnt         	; bit counter
        	clrf    rxreg          		; clear input buffer
		nop_2				; adjust
rx_c		movlw	(baud_div - 12)/4
		call    delay         		; delay full data period  (center data window)
   		clrc		        	; start at known state
        	btfsc   PORTB,rs232_inp		; incoming data 
        	setc				; 
        	rrf     rxreg,f         	; rotate pin status into receive buffer lsb first
        	decfsz  bit_cnt,f      		; any more bits?
        	goto    rx_c            	; yep, start over

		movlw	(baud_div - 4)/4	; not too critical
        	call    delay           	; put us into the stop bit
		movfw	rxreg			; stuff rsdata into w for the return
		clrc				; good data
		return                  	; and return

;******************************************************************
;routine converts 0 - F ascii to the low nibble, trashes r0
; c=0 if all went well, c=1 if incoming w was out of range
; note we only handle upper case
;******************************************************************
asc_hex		movwf	r0		; save it

		movlw	0x30 & 0xff
		subwf	r0,w		
		skpc
		goto	asc_err
		
		movlw	0x3A & 0xff
		subwf	r0,w		
		skpc
		goto	numbers		; is 0-9 ascii
		
		movlw	0x41 & 0xff
		subwf	r0,w		
		skpc	
		goto	asc_err

        	movlw	0x47 & 0xff
		subwf	r0,w		
		skpc
		goto	letters		; is A-F ascii

asc_err		setc			; is an error, input out of range
		return			; set carry and exit


numbers 	movlw   '0'            	;subtract 30hex from the ascii
        	goto    asc_1           ;and voila instant number, exit
letters 	movlw   0x37            ;subtract 37hex from the ascii
asc_1   	subwf   r0,w        	;do the actual subtraction and stuff results back into w 
 		clrc			; all's well
       		return          	; and exit

;******************************************************************
;routine converts w into two ascii char and ships them, hi nibble first
;******************************************************************
send_asc	movwf	r1		; save w 
		swapf	r1,w		; get high nibble first
		call	hex_asc		; convert to ascii
		call	rs_tx
		movfw	r1		; get lo nibble
		call	hex_asc
		call	rs_tx
		movfw	r1		; restore w
		return

;******************************************************************
;routine converts 0-f hex in w to 0 to F ascii
;******************************************************************
hex_asc	    	andlw   b'00001111'     ;lower nybble only
    		addlw   .6              ;0-9 or A-F?
    		skpndc                  ;dc==1 if larger than 9
      		addlw	.7              ;adjust A-F
    		addlw   (0x30-6)        ;add offset to convert to ascii
		return

;***************************************************************************
; Transmit serial data.  info in w is clocked out 8n1 
;***************************************************************************
rs_tx   movwf   txreg           ;xmit w
        movlw   .8
        movwf   bit_cnt         ;   	eight bits to send
        bcf     PORTB,rs232_out ;  	start bit
	movlw	(baud_div - 5)/4;1  	bit period 	
        call    delay		;2
rs_1    rrf     txreg,f         ;1    	rotate bit to send into carry lsb first
        skpc		        ;1   	test carry and set output pin accordingly
        bcf     PORTB,rs232_out ;1    	no false edges
        skpnc			;1
        bsf     PORTB,rs232_out	;1

	movlw	(baud_div - 13)/4	;1	
        call    delay           ;2	delay a bit period

        decfsz  bit_cnt,f            ;1	any more to send?
        goto    rs_1		;2

        bsf     PORTB,rs232_out ;	stop bit
	movlw	(baud_div - 4)/4; 	not too critical	
        call    delay
   	return                  ; 	and return


;******************************************************************
; read eeprom 
; in w=addr / out with w=data   / incs EEADR
;******************************************************************
rd_ee_w movwf   EEADR           ;EEADR = w
rd_ee   bank1		      	;select bank 1
        bsf     EECON1,RD       ;read EEPROM
        bank0      		;select bank 0
        movfw   EEDATA        	;get EEDATA into w
	incf	EEADR
        return

;******************************************************************
; program the eeprom
; eedata goes into eeadr      / incs EEADR
;******************************************************************
wr_ee_w	movwf	EEDATA		; stub to store w	
wr_ee   bank1     		;select bank 1
        bsf     EECON1,WREN     ;enable EEPROM write
        movlw   h'55'
        movwf   EECON2          ;write 55
        movlw   h'AA'
        movwf   EECON2          ;write AA
        bsf     EECON1,WR       ;start write
        bcf     EECON1,WREN     ;disable write
EEPLoop
        nop
        btfsc   EECON1,WR       ;is write cycle complete?
        goto    EEPLoop         ;wait for write to finish
        bank0			;select bank 0
	incf	EEADR	
        return

;********************************************************************
;* general purpose delay routine delay (4*w) + 4 overhead                                            *
;********************************************************************
delay   addlw	-1	        ;1 stuff into working register
     	skpz	        	;1/2
        goto    delay           ;2
        return

;------------------- tic delay --------------------------------------
; delay w * number of isr tics
tic_delay	movwf 	r0			; save in  a temp reg
		movfw 	tic			; get current tick
		addwf 	r0,f			; (.000108509 @3.6864mhz)   
				
td_1		movfw 	tic			; keep looping until tics = (incoming tics + w) 
		xorwf 	r0,w
		skpz		
		goto	td_1			; and wait
		return

;---------------------------------------------------------------
;********************* computed tables ************************* 
;---------------------------------------------------------------
		org 0x3c0

;------------copyright message ------------
;
get_data	bsf PCLATH,0
		bsf PCLATH,1
		movwf PCL

copy_r		dt cr,lf	; cr lf
		dt " PIC POCSAG ENCODER" 
		dt cr,lf
		dt "COPYRIGHT Henry Carl Ott"
		dt cr,lf
		dt " HIT ANY KEY TO PROGRAM"
		dt cr,lf
		dt 0		; eod

		end
