/*
nullmodem, based on Frederic RIBLE's kissnetd
*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <unistd.h>
#include <assert.h>
#include <sys/file.h>
#include <string.h>
#include <errno.h>
#include <syslog.h>
#include <time.h>
#include <iostream>
#include <termios.h>
#include <stdlib.h>
#include <fcntl.h>
#include <pty.h>
#include <sched.h>
#include <sys/mman.h>
#include <string>
#include <vector>

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif
#include <getopt.h>

using namespace std;

static int VerboseMode = 5; // 0 - Quiet, 5 - Informative, 9 Debugging.
static bool nonblock=true;

// Usage
// if (verbLevel(5)) {
//  output informative stuff
// }
//
// if (verbLevel(9) {
//   output chatty debug stuff
// }
#define verbLevel(n) ((n) <= VerboseMode)

void logMessage(const string& message, int terrno)
{
  syslog(LOG_WARNING, "nullmodem : %s : %s\n",
         message.c_str(), strerror(terrno));

  if (verbLevel(5))
  {
    cerr << message << " : " << strerror(terrno) << endl;
  }
}




class PortDescriptor
{
private:
  string outputBuffer;
  int lockFd;
  uid_t uid;
  gid_t gid;

public :
  int fd;
  string name;

  PortDescriptor(const string& _name, const uid_t _uid = 0, const gid_t _gid = 0);

  // file locking  $name.lock
  void getLock();
  void dropLock();

  void reopen(); // (re)open  file descriptor
  bool needToReopen(); // in other words: file descriptor in invalid!

  void doWrite();  // write data from outputBuffer to the file descriptor
  bool hasDataToWrite() const; // outputBuffer is not empty
  void write(const string& buffer); // write data to the outputBuffer

  bool read(string& buffer); // read data into &buffer
};




PortDescriptor::PortDescriptor(const string& _name, const uid_t _uid, const gid_t _gid) : lockFd( -1), fd( -1), name(_name), gid(_gid), uid(_uid)
{
  if (verbLevel(5))
    cout << "Opening port " << name << endl;
}


bool PortDescriptor::hasDataToWrite() const
{
  return !outputBuffer.empty();
}


bool PortDescriptor::needToReopen()
{
  return (fd < 0);
}


void PortDescriptor::getLock()
{
  string lockFile(name);
  lockFile += ".lock"; // Create an additional lock file, I'm not sure how opening symbolic links directly would behave.

  lockFd = open(lockFile.c_str(), O_CREAT, 0770);
  if (lockFd < 0)
  {
    logMessage("Couldn't open/creat lock file, exiting now.", errno);
    exit( -1);
  }

  int result = flock(lockFd, LOCK_EX | LOCK_NB);
  if (result < 0)
  {
    logMessage("Couldn't get lock, nullmodem already running.", errno);
    exit( -1);
  }
}


void PortDescriptor::dropLock()
{
  if (lockFd < 0)
    return ;

  int result = flock(lockFd, LOCK_UN );
  if (result < 0)
  {
    logMessage("Dropping lock failed.", errno);
  }

  close(lockFd);
  lockFd = -1;
}


void PortDescriptor::reopen()
{
  int slavefd;
  int masterfd;

  getLock();

  int ret = openpty(&masterfd, &slavefd, 0, 0, 0);
  if (ret < 0)
  {
    logMessage("Open pty failed", errno);
    dropLock();
    return ;
  }

  if (verbLevel(9))
  {
    cout << "openpty ret=" << ret << endl;

    cout << "masterfd=" << masterfd << endl;
    cout << "slavefd=" << slavefd << endl;

    cout << "master=" << ttyname(masterfd) << endl;
    cout << "slave=" << ttyname(slavefd) << endl;
  }

  struct termios tm;
  memset(&tm, 0, sizeof(tm));
  tm.c_cflag = CS8 | CREAD | CLOCAL;
  if (tcsetattr(masterfd, TCSANOW, &tm))
    cerr << "master: tcsetattr" << endl;
  memset(&tm, 0, sizeof(tm));
  tm.c_cflag = CS8 | CREAD | CLOCAL;
  if (tcsetattr(slavefd, TCSANOW, &tm))
    cerr << "slave: tcsetattr" << endl;
  if (true==nonblock)
  {
    if (verbLevel(9))
    {
      cout << "seting NONBLOCKing io" << endl;
    }
    fcntl(masterfd, F_SETFL, fcntl(masterfd, F_GETFL, 0) | O_NONBLOCK);
  }
  else
  {
    if (verbLevel(9))
    {
      cout << "seting BLOCKing io (i.e. doing nothing)" << endl;
    }
  }

  unlink(name.c_str());
  symlink(ttyname(slavefd), name.c_str());

  if (0 != uid)
  {
    if (chown(ttyname(slavefd), uid, gid) < 0)
    {
      logMessage("chown failed", errno);
    }
    if (chown(name.c_str(), uid, gid) < 0)
    {
      logMessage("chown failed", errno);
    }
  }
  fd = masterfd;
}


void PortDescriptor::doWrite()
{
  int result = ::write(fd, outputBuffer.c_str(), outputBuffer.size());
  if (result < 0)
  {
    logMessage("Write failed", errno);
  }
  if (result <= 0)
  {
    close(fd);
    fd = -1;
    dropLock();
    return ;
  }
  outputBuffer.erase(0, result);
}


bool PortDescriptor::read(string& buffer)
{
  int length;
  const size_t bufSize = 65535;
  char buf[bufSize];

  length = ::read(fd, &buf, bufSize);
  if (length < 0)
  {
    logMessage("Read failed ", errno);
  }
  if (length <= 0)
  {
    close(fd);
    fd = -1;
    dropLock();
    return false;
  }
  if (verbLevel(9))
  {
    cerr << "Read port " << name
         << " : rc=" << length
         << endl;
  }

  buffer.assign(buf, length);
  return true;
}


void PortDescriptor::write(const string& buffer)
{
  outputBuffer += buffer;
}




class PortFactory
{
private:
  typedef vector<PortDescriptor*> portList_t;
  portList_t portList;

public :
  PortFactory();

  PortDescriptor * createPort(const string& name, const uid_t uid, const gid_t gid);

  // only called at startup to open all ports
  void tickReopen();

  // write buffer to all PortDescriptor::outputBuffer except for originator
  void broadcast(const string& buffer, PortDescriptor * originator);

  void eventLoop();
};


PortFactory::PortFactory()
{}


PortDescriptor * PortFactory::createPort(const string& name, const uid_t uid, const gid_t gid)
{
  PortDescriptor * result = new PortDescriptor(name, uid, gid);

  portList.push_back(result);
  return result;
}


void PortFactory::tickReopen()
{
  for (size_t i = 0; i < portList.size(); i++)
  {
    PortDescriptor& port = *portList[i];

    if (port.needToReopen())
    {
      if (verbLevel(5))
      {
        cout << "Reopening, " << port.name << ", port " << i << endl;
      }
      port.reopen();
    }
  }
}


void PortFactory::broadcast(const string& buffer, PortDescriptor * originator)
{
  if (verbLevel(9))
  {
    cout << "data= '" << buffer << "'" << endl;
  }

  for (portList_t::iterator i = portList.begin();
       i != portList.end(); i++)
  {
    PortDescriptor * port = *i;

    if (port == originator)
      continue;

    port->write(buffer);
  }
}


void PortFactory::eventLoop()
{
  fd_set readFdSet;
  fd_set writeFdSet;
  int rc;
  struct timeval timeout;

  while (true)
  {
    timeout.tv_sec = 30;
    timeout.tv_usec = 0;

    FD_ZERO(&readFdSet);
    FD_ZERO(&writeFdSet);
    for (portList_t::const_iterator i = portList.begin();
         i != portList.end(); i++)
    {
      PortDescriptor& port = **i;
      if (port.fd < 0)
      {
        port.reopen();
      }
      if (port.fd >= 0)
      {
        FD_SET(port.fd, &readFdSet);
        if (port.hasDataToWrite())
          FD_SET(port.fd, &writeFdSet);
      }
    }

    rc = select(FD_SETSIZE, &readFdSet, &writeFdSet, 0, &timeout);

    if (rc < 0)
    {
      logMessage("Error in select", errno);
      return ;
    }
    if (0 == rc)
    { // Timed out, loop around and try again.
      if (verbLevel(9))
      {
        cerr << "Timed out, trying again." << endl;
      }
      continue;
    }

    // read data and fill the output buffers
    for (portList_t::iterator i = portList.begin();
         i != portList.end(); i++)
    {
      PortDescriptor& port = **i;
      if (FD_ISSET(port.fd, &readFdSet))
      {
        string buffer;
        if (port.read(buffer))
        {
          broadcast(buffer, &port);
        }
      }
    }
    // write data form all output buffers
    for (portList_t::iterator i = portList.begin();
         i != portList.end(); i++)
    {
      PortDescriptor& port = **i;
      if (FD_ISSET(port.fd, &writeFdSet))
      {
        port.doWrite();
      }
    }
  } // end of while (true)
}




void printHelpAndExit(const char * plaint = "Usage:-")
{
  cout << plaint << endl
       << endl
       << "nullmodem [-F] [-V] [-q] [-r] [-m] tty..." << endl
       << endl
       << "DESCRIPTION" << endl
       << "nullmodem  creates  a  virtual network of Pseudo-Terminals." << endl
       << "It can be used as an adapter to connect two programs" << endl
       << "that normally need serial interface cards." << endl
       << endl
       << "OPTIONS" << endl
       << "-F, --foreground" << endl
       << "Unlike normal operation, nullmodem will not detach from the terminal" << endl
       << "when given this option. Some  debugging  informations will be visible." << endl
       << endl
       << "-q, --quiet" << endl
       << "Start quietly - for init scripts." << endl
       << endl
       << "-v, --verbose" << endl
       << "Print debug info." << endl
       << endl
       << "-V, --version" << endl
       << "Print the version number and exit." << endl
       << endl
       << "-r, --realtime" << endl
       << "Start with realtime priority (i.e. use the round-robin scheduler)." << endl
       << "This requires nullmodem to run with root privileges." << endl
       << endl
       << "-m, --memorylock" << endl
       << "Lock the executable of nullmodem to RAM. This keeps nullmodem from" << endl
       << "being swapped out." << endl
       << endl
       << "--setuid n" << endl
       << "Set the UserID (UID) of the pseudo terminal spacial files to n." << endl
       << endl
       << "--setgid n" << endl
       << "Set the GroupID (GID) of the pseudo terminals spacial files to n." << endl
       << endl
       << "--blocking" << endl
       << "Use blocking IO to handle the interfaces (default: NONBLOCKing)." << endl
       << endl;
  exit(1);
}


int main(int argc, char *argv[])
{

  bool lockmem = false;
  bool schedrr = false;
  bool daemonize = true;
  uid_t uid = 0;
  gid_t gid = 0;

  static const struct option long_options[] =
  {
    { "realtime", no_argument, 0, 'r'
    },
    { "memorylock", no_argument, 0, 'm'},
    { "foreground", no_argument, 0, 'F' },
    { "quiet", no_argument, 0, 'q' },
    { "verbose", no_argument, 0, 'v' },
    { "version", no_argument, 0, 'V' },
    { "help", no_argument, 0, 'h'},
    { "setuid", required_argument, 0, 1000},
    { "setgid", required_argument, 0, 1001},
    { "blocking", no_argument, 0, 1002},
    { 0, 0, 0, 0 }
  };

  int c;
  while ((c = getopt_long(argc, argv, "rmFqvVh?", long_options, NULL)) != -1)
  {
    switch (c)
    {
    case 'r':
      schedrr = true;
      break;
    case 'm':
      lockmem = true;
      break;
    case 'F':
      daemonize = false;
      break;
    case 'q':
      VerboseMode = 0;
      break;
    case 'v':
      VerboseMode = 9;
      break;
    case 'V':
      cout << argv[0] << " " << VERSION
           << " (compiled " __DATE__ " " __TIME__ ")"
           << endl;
      exit(0);
      break;
    case 'h' :
    case '?' :
      printHelpAndExit();
      break;
    case 1000:
      uid = atoi(optarg);
      break;
    case 1001:
      gid = atoi(optarg);
      break;
    case 1002:
      nonblock = false;
      break;
    default:
      printHelpAndExit("No such option");
    }
  }

  if (verbLevel(5))
    cerr << argv[0] << " " VERSION " (compiled " __DATE__ " " __TIME__ ")" << endl;

  if ( optind + 2 > argc)
  {
    printHelpAndExit("Too few arguments. Give at least two arbitrary names for your pseudo tty endings... ");
  }

  PortFactory portFactory;
  int argIdx = optind;
  while (argv[argIdx])
  {
    portFactory.createPort(argv[argIdx++], uid, gid);
  }

  portFactory.tickReopen();

  if (daemonize)
  {
    if (daemon(0, 0) < 0)
    {
      logMessage("Daemonize failed", errno);
    }
    umask(0);
  }

  if (lockmem)
  {
    if (mlockall(MCL_CURRENT | MCL_FUTURE) == -1)
      logMessage("Error in mlockall", errno);
  }

  if (schedrr)
  {
    struct sched_param schp;
    memset(&schp, 0, sizeof(schp));
    schp.sched_priority = sched_get_priority_min(SCHED_RR) + 1;
    if (sched_setscheduler(getpid(), SCHED_RR, &schp) != 0)
      logMessage("Error in sched_setscheduler ", errno);
  }

  portFactory.eventLoop();

  return 0;
}
