#!/usr/bin/perl
#
#
#
## modules
use strict;
use DBI;
use Time::HiRes;
use URI::Escape;
use Text::Iconv;
use XML::Simple;
use Crypt::IDEA;
use Digest::MD5;
use Date::Parse;
use Log::LogLite;
use Email::Simple;
use IPC::ShareLite qw(:lock);
use LWP::UserAgent;
use IO::Socket::INET;
use Net::SMTP::Server;
use Storable qw(freeze thaw);
use Net::SMTP::Server::Client;

## configuration ##
# config file
my $config = "filterd.cfg";

# logging
my $log_level = 6;
my $log_file = "filterd.log";
my $log_err_file = "filterd_error.log";
my $log = new Log::LogLite($log_file, $log_level);

# pid file
my $pid_file = "filterd.pid";

# smtp server settings
my $smtp_listen = "0.0.0.0";
my $smtp_port = "2526";

# message lockout period in sec
my $msg_lockout = 60;

# purge messages from messages array after n seconds
my $msg_purge = 120;

# iconv config
my $isotoutf = Text::Iconv->new("LATIN1", "UTF-8");
my $utftoiso = Text::Iconv->new("UTF-8", "LATIN1");

# mysql databse
my $db_host = "localhost";
my $db_port = "3306";
my $db_user = "filterd";
my $db_pass = "pass";
my $db_dbase = "filterd";
my $db_pocsag_table = "messages";
my $db_mapping_table = "mappings";
my $db_function_table = "functions";
my $db_status_table = "status";

# sms api - set host priority with hash index
my %sms_config = ( 
  1 => { type => "mobilant", # should be compatible with smstrade
         key => "key",
         route => "direct",
       },

  2 => { type => "kannel",
         host => "192.168.1.1",
         path => "/api",
         port => "443",
         mode => "https",
         username => "user",
         password => "pass",
       },

  3 => { type => "sms77",
         username => "user",
         password => "pass",
         msg_type => "quality",
       },
);

# push api
my $prowl_provider_key = "key";
my $nma_developer_key = "key";
my $push_app_name = "PDW-Notification";
my $push_default_priority = 2;
## configuration ##

## shm handling ##
my $shm = IPC::ShareLite->new(
  -key     => 4503,
  -mode    => 0644,
  -create  => 'yes',
  -destroy => 'yes'
) or die "Cannot create SHM: $!\n";
my %empty;
$shm->store(freeze(\%empty));
## shm handling ##

## functions ##
# decrypt IDEA with given key
sub decryptIDEA {
  my $message = $_[0];
  my $key = pack("H32", $_[1]);
  my $cipher = new IDEA $key;

  return unpack("H16", $cipher->decrypt($message)); # NB - 8 bytes
}

# sends sms over sms api (kannel or sms77)
sub sendSMS {
  my $recipient = uri_escape($_[0]);
  my $message = uri_escape($_[1]);

  my $lwp = LWP::UserAgent->new;
  $lwp->timeout(5);
  $lwp->env_proxy;

  # go trough all hosts, if any accepts the message, return the function
  foreach my $idx (sort(keys %sms_config)) {
    my $type = $sms_config{$idx}{"type"};

    if ($sms_config{$idx}{"type"} eq "kannel") {
      my $host = $sms_config{$idx}{"host"};
      my $username = $sms_config{$idx}{"username"};
      my $password = $sms_config{$idx}{"password"};
      my $path = $sms_config{$idx}{"path"};
      my $port = $sms_config{$idx}{"port"};
      my $mode = $sms_config{$idx}{"mode"};

      my $response = $lwp->get($mode.'://'.$host.':'.$port.$path.'?username='.$username.'&password='.$password.'&to='.$recipient.'&text='.$message.'&charset=UTF-8');

      if (($response->is_success) && ($response->code == 202)) {
        $log->write("SMS API (kannel) on host ".$host." returns success:'".$response->decoded_content."'");
        return 1;
      } else {
        $log->write("SMS API (kannel) on host ".$host." returns error:'".$response->status_line."'");
      }
    }
    elsif ($sms_config{$idx}{"type"} eq "mobilant") {
      my $key = $sms_config{$idx}{"key"};
      my $route = $sms_config{$idx}{"route"};

      my $response = $lwp->get("https://gw.mobilant.net/?key=".$key."&to=".$recipient."&message=".$message."&route=".$route."&concat=1&charset=UTF-8");
      if (($response->is_success) && ($response->code == 200) && ($response->decoded_content eq "100")) {
        $log->write("SMS API (mobilant) returns success:'".$response->decoded_content."'");
        return 1;
      } else {
        $log->write("SMS API (mobilant) returns error:'".$response->decoded_content."'");
      }
    }
    elsif ($sms_config{$idx}{"type"} eq "sms77") {
      my $username = $sms_config{$idx}{"username"};
      my $password = $sms_config{$idx}{"password"};
      my $msg_type = uri_escape($sms_config{$idx}{"msg_type"});
      my $msg_from = uri_escape($sms_config{$idx}{"msg_from"});

      my $response = $lwp->get('https://gateway.sms77.de/?u='.$username.'&p='.$password.'&to='.$recipient.'&text='.$message.'&type='.$msg_type);
      if (($response->is_success) && ($response->code == 200) && ($response->decoded_content eq "100")) {
        $log->write("SMS API (sms77) returns success:'".$response->decoded_content."'");
        return 1;
      } else {
        $log->write("SMS API (sms77) returns error:'".$response->decoded_content."'");
      }
    }
  }
  return 0;
}

# sends push notifications over prowl and nma api
sub sendPush {
  my $apikey = $_[0];
  my $event = $_[1];
  my $message = $_[2];
  my $priority = $_[3];
  my $key_len = length($apikey);

  my $xs = XML::Simple->new;
  my $lwp = LWP::UserAgent->new;
  $lwp->timeout(10);
  $lwp->env_proxy;

  my $api_url;
  my $response;
  my $provider;

  # prowl has 40 bytes api keys
  if ($key_len == 40) {
    $provider = "Prowl";
    $api_url = "https://api.prowlapp.com/publicapi/add";
    $response = $lwp->post($api_url, [ apikey => $apikey,
                                       application => $push_app_name,
                                       event => $event,
                                       description => $message,
                                       priority => $priority,
                                       providerkey => $prowl_provider_key
                                     ]);
  }
  # nma has 48 bytes api keys
  elsif ($key_len == 48) {
    $provider = "NMA";
    $api_url = "https://www.notifymyandroid.com/publicapi/notify";
    $response = $lwp->post($api_url, [ apikey => $apikey,
                                       application => $push_app_name,
                                       event => $event,
                                       description => $message,
                                       priority => $priority,
                                       developerkey => $nma_developer_key
                                     ]);
  }

  if (($response->is_success) && ($response->code == 200)) {
    my $ref = $xs->XMLin($response->decoded_content);
    $log->write("Push API (".$provider.") returns success. remaining messages:".$ref->{'success'}->{'remaining'});
    return 1
  }
  else {
    $log->write("Push API (".$provider.") returns error:'".$response->status_line."'");
    return 0;
  }
}

# handle notifications
sub handleNotifications {
  my $ric = $_[0];
  my $ric_desc = $_[1];
  my $subric = $_[2];
  my $subric_desc = $_[3];
  my $text = $_[4];
  my $text_len = $_[5];

  # read config
  { package Config; do $config };
  my %ric_to_message = %Config::ric_to_message;
  my %push_prio = %Config::push_prio;
  my %message_len_filter = %Config::message_len_filter;
  my %subric_filter = %Config::subric_filter;

  # send sms / prowl notifications
  if (defined($ric_to_message{$ric})) {
    foreach (@{$ric_to_message{$ric}}) {
      # subric filter (skip subrics if configured in subric_filter hash)
      if(defined($subric_filter{$_}{$ric}{$subric}) && !$subric_filter{$_}{$ric}{$subric}) {
        $log->write("Skip notification to:'".$_."' for RIC:".$ric." Sub:".$subric." disabled for recipient");
        next;
      }

      # message length filter
      if (defined($message_len_filter{$_}{$ric}) && ($text_len < $message_len_filter{$_}{$ric}) && $text_len) {
        $log->write("Skip notification to:'".$_."' for RIC:".$ric." message to short (".$text_len." vs ".$message_len_filter{$_}{$ric}.")");
        next;
      }

      # fork child to get faster message processing
      my $pid = fork();
      if (!defined($pid)) {
        $log->write("Cannot fork new process for notifcation handling: $!");
      }
      elsif (!$pid) {
        # number triggers sms, text triggers push
        my $message;
        my $event;
        if ($_ =~ /^\d+$/) {
          # add ric and subric description (SMS)
          if (!$ric_desc) {
            if (!$subric_desc) {
              $message = "RIC ".$ric.": ".$text;
            } else {
              $message = "RIC ".$ric." (".$subric_desc."): ".$text;
            }
          } else {
            if (!$subric_desc) {
              $message = $ric_desc.": ".$text;
            } else {
              $message = $ric_desc." (".$subric_desc."): ".$text;
            }
          }
          $log->write("Sending SMS notification for RIC:".$ric." to:'".$_."'");
          sendSMS($_, $message);
        } else {
          # add ric and subric description (Push)
          if (!$ric_desc) {
            if (!$subric_desc) {
              $event = "RIC ".$ric;
            } else {
              $event = "RIC ".$ric." (".$subric_desc.")";
            }
          } else {
            if (!$subric_desc) {
              $event = $ric_desc;
            } else {
              $event = $ric_desc." (".$subric_desc.")";
            }
          }
          # push priority override
          my $prio = $push_default_priority;
          if (defined($push_prio{$_}{$ric})) {
            $prio = $push_prio{$_}{$ric};
          }
          $log->write("Sending Push notification for RIC:".$ric." to:'".$_."' with priority:".$prio);
          sendPush($_, $event, $text, $prio);
        }
        # ensure the child exits correctly
        exit 0;
      }
    }
  }
  return 1;
}

# get ric description from database
sub getRicDescription {
  my $dbh = $_[0];
  my $ric = $_[1];

  my $ric_desc = 0;
  my $sth = $dbh->prepare("SELECT realname FROM ".$dbh->quote_identifier($db_mapping_table)." WHERE ric = ".$dbh->quote($ric));
  $sth->execute();
  while (my $ref = $sth->fetchrow_hashref()) {
    $ric_desc = $isotoutf->convert($ref->{'realname'});
  }
  $sth->finish();
  return $ric_desc;
}

# get ric function description from database
sub getRicFunctionDesc {
  my $dbh = $_[0];
  my $sub = $_[1];

  my $sub_desc = 0;
  my $sth = $dbh->prepare("SELECT description FROM ".$dbh->quote_identifier($db_function_table)." WHERE sub = ".$dbh->quote($sub));
  $sth->execute();
  while (my $ref = $sth->fetchrow_hashref()) {
    $sub_desc = $isotoutf->convert($ref->{'description'});
  }
  $sth->finish();
  return $sub_desc;
}

# insert message into database
sub messageToDb {
  my $dbh = $_[0];
  my $ric = $_[1];
  my $subric = $_[2];
  my $text = $_[3];
  my $ts = $_[4];

  $dbh->do("INSERT INTO ".$dbh->quote_identifier($db_pocsag_table)." (datetime,ric,sub,message) VALUES (FROM_UNIXTIME(".$ts."),".$dbh->quote($ric).",".$dbh->quote($subric).",".$dbh->quote($utftoiso->convert($text)).");");
  return 1;
}

# insert status message into database
sub statusToDb {
  my $dbh = $_[0];
  my $ric = $_[1];
  my $ts = $_[2];

  my $sth = $dbh->prepare("SELECT received FROM ".$dbh->quote_identifier($db_status_table)." WHERE ric = ".$dbh->quote($ric));
  $sth->execute();
  if (!$sth->rows) {
    $dbh->do("INSERT INTO ".$dbh->quote_identifier($db_status_table)." (received,ric) VALUES (FROM_UNIXTIME(".$ts."),".$dbh->quote($ric).");");
  } else {
    $dbh->do("UPDATE ".$dbh->quote_identifier($db_status_table)." SET received = FROM_UNIXTIME(".$ts.") WHERE ric = ".$dbh->quote($ric).";");
  }
  $sth->finish();
  return 1;
}

# process the pdw messages
sub processMsg {
  my $client = $_[0];
  my $dbh = $_[1];

  # maps pdw function bits
  my %function_map = ( "1" => "A", "2" => "B", "3" => "C", "4" => "D" );

  # read config
  { package Config; do $config };
  my %special_ric = %Config::special_ric;
  my %ric_encryption = %Config::ric_encryption;
  my $regex_filter = $Config::regex_filter;
  my %express_alarm = %Config::express_alarm;

  # parse email body
  my $email = Email::Simple->new($client->{MSG});

  # open md5 digest object
  my $md5 = Digest::MD5->new;

  # extract subject - take care of PSW settings
  my $subject = $email->header("Subject");

  # parse data from subject
  my @data = split(" ", $subject);
  my $ric = @data[0];
  my $time = @data[1];
  my $date = @data[2];
  my $system = @data[3];
  my $type = @data[4];

  # current unix timestamp
  my $now = time();

  # parse message timestamp and use as "message_ts", so we take care of network latency
  my $message_ts = str2time($date." ".$time);
  # fallback to time() if date was unparsable or in the future. NOTE: use US date format in pdw
  if (!defined($message_ts) || ($message_ts >= $now + (60 * 60 * 24))) {
    $log->write("Problem parsing date:".$date." time:".$time." with parsedate() switching to time()");
    $message_ts = $now;
  }

  # skip non pocsag messages
  if ($system !~ /POCSAG/) {
    $log->write("We support only POCSAG, skipping message");
  }

  # last char of pdw mode was function bit (pocsag), the rest was the system description
  my @sys_data = split("-", $system);
  my $system = @sys_data[0];
  my $subric = $function_map{@sys_data[1]};

  # find type in string and extract message from pdw message
  my $msg_idx;
  while ($subject =~ /$type/g) {
    $msg_idx = pos($subject) + 1;
  }
  my $text = substr($subject, $msg_idx);

  # remove unneeded whitespaces
  $text =~ s/^\s+|\s+$//g;

  # handle tone only messages (also not correct parsed messages)
  my $text_len = length($text);
  if ($text_len == 0 || ($type =~ /NUMERIC/ && ($text =~ /TONE\ ONLY/ || $text eq "02000"))) {
    $text = "[Nur-Ton]";
    $text_len = 0;
  }

  # add text to md5 ctx and build hex hash
  $md5->add($text);
  my $text_md5 = $md5->hexdigest;

  # encode message correctly
  $text = $isotoutf->convert($text);

  # log message from pdw
  $log->write("RIC:".$ric." Time:".$time." Date:".$date." System:".$system." Sub:".$subric." Type:".$type." Text:'".$text."'"); 

  # handle and filter special rics
  if (defined($special_ric{$ric})) {
    if (!$special_ric{$ric}) {
      $log->write("RIC:".$ric." defined as special RIC but configured to pass filter");
    }
    elsif ($special_ric{$ric} == 1) {
      $log->write("Filtered RIC:".$ric);
      return 1;
    }
    elsif ($special_ric{$ric} == 2) {
      $log->write("Filtered RIC:".$ric." (status ric)");
      statusToDb($dbh, $ric, $message_ts);
      return 1;
    }
    elsif ($special_ric{$ric} == 3) {
      $log->write("Passing RIC:".$ric." (delimiter ric)");
    }
  }

  # filter rics with regex from config (let pass delimiter rics)
  if (($ric !~ /$regex_filter/) && !defined($special_ric{$ric})) {
    $log->write("Filtered RIC:".$ric." (regex:'".$regex_filter."')");
    return 1;
  }

  # decrypt message if IDEA key exists for ric
  if (defined($ric_encryption{$ric})) {
    $log->write("Found IDEA Key for RIC:".$ric.". Calling decryptIDEA");
    $text = decryptIDEA($text, $ric_encryption{$ric});
  }

  # get messages array from shm and try to lock shm or wait until we can access data
  while(1) {
    if ($shm->lock(LOCK_SH|LOCK_NB)) {
      last;
    }
    else {
      usleep(50);
    }
  }
  my %messages = %{thaw($shm->fetch)};

  # cleanup messages array
  foreach (keys %messages) {
    if ($messages{$_}[0] + $msg_purge <= $message_ts) {
      delete($messages{$_});
      next;
    }
  }

  # lockout duplicate messages (let ea text and special rics pass)
  if(defined($messages{$ric}) && !defined($express_alarm{$ric}) && !defined($special_ric{$ric})) {
    if (($messages{$ric}[0] + $msg_lockout >= $message_ts) && ($messages{$ric}[1] eq $subric) && ($messages{$ric}[3] eq $text_md5)) {
      $log->write("Lockout duplicate RIC:".$ric);
      return 1;
    }
  }

  # get subric description from db
  my $subric_desc = getRicFunctionDesc($dbh, $subric);
  # change text if we have tone only messages and delete description to avoid duplicate informations in sms and push messages
  if ($subric_desc && !$text_len) {
    $text = $subric_desc;
    $subric_desc = 0;
  }

  # ea (express alarm) handling
  if (defined($express_alarm{$ric})) {
    $log->write("EA: detected RIC:".$ric." Sub:".$subric);
    # tone only ea ric makes no sense
    if ($text_len) {
      # get delimiter ric timestamp from messages hash
      $log->write("EA: found last delimiter RIC:".$express_alarm{$ric}." with ts:".$messages{$express_alarm{$ric}}[0]);
      foreach (keys %messages) {
        # skip special or ea text rics
        next if (defined($special_ric{$_}) || defined($express_alarm{$_}));
        # message was in delimiter period
        if ($messages{$_}[0] >= $messages{$express_alarm{$ric}}[0]) {
          # message subric was equal and the message was tone only
          if (($messages{$_}[1] eq $subric) && !$messages{$_}[2]) {
            $log->write("EA: renotify RIC:".$_." Sub:".$subric);
            # handle (re)notifications
            handleNotifications($_, getRicDescription($dbh, $_), $subric, $subric_desc, $text, $text_len);
            # (re)add message to database
            messageToDb($dbh, $_, $subric, $text, $messages{$_}[0]);
          } else {
            $log->write("EA: found real RIC:".$_." in EA period but either subric was wrong or RIC was not tone only");
          }
        }
      }
    } else {
      $log->write("EA: tone only EA RIC makes no sense");
    }
  }
  elsif (!defined($special_ric{$ric})) { # skip adding special rics to database or try to notify
    # handle notifications
    handleNotifications($ric, getRicDescription($dbh, $ric), $subric, $subric_desc, $text, $text_len);
    # add message to database (skip ea rics)
    messageToDb($dbh, $ric, $subric, $text, $message_ts);
  }

  # add message timestamp, subric, length and checksum to messages array
  # needed for message lockout and ea alarm detection and store this array in shm
  @{$messages{$ric}} = ($message_ts, $subric, $text_len, $text_md5);
  $shm->store(freeze(\%messages));
  $shm->unlock;

  return 1;
}
## functions ##

## main ##
# we dont wait for our childs and dont want zombies
$SIG{CHLD} = 'IGNORE'; 

# redirect STDOUT and STDERR to error log file
open(STDOUT, ">>".$log_err_file);
open(STDERR, ">>".$log_err_file);

# log startup
$log->write("Starting filterd");

# write pid file
open(PID_FILE, '>'.$pid_file);
print PID_FILE $$;
close(PID_FILE);

# connect to mysql db
my $dsn = "DBI:mysql:database=$db_dbase;host=$db_host;port=$db_port";
my $dbh = DBI->connect($dsn, $db_user, $db_pass, { AutoCommit => 1, RaiseError => 0 } ) || die("cannot connect to mysql db: $!\n");
$dbh->{mysql_auto_reconnect} = 1;

# open smtp Server to receive messages from PSW - take care of PSW settings
my $server = new Net::SMTP::Server($smtp_listen, $smtp_port) || die("Cannot start smtp server: $!\n");
$log->write("SMTP server started on port:".$smtp_port);

# catch sigint and sigterm
sub catch_exit {
  # close database connection
  $dbh->disconnect();

  # remove pid file
  unlink($pid_file);
  exit;
}
$SIG{INT} = \&catch_exit;
$SIG{TERM} = \&catch_exit;

while(my $conn = $server->accept()) {
  # process smtp client connection
  my $client = new Net::SMTP::Server::Client($conn);
  my $remote_host = $client->{SOCK}->peerhost();

  my $pid = fork();
  if ($pid) {
    $log->write("New connection from:".$remote_host." child pid:".$pid);
  }
  elsif (!$pid) {
    # block until the client completes the SMTP transaction
    $client->process || exit 0;

    # accept only mails from pdw
    if ($client->{FROM} eq "<pdw>") {
      # process message
      processMsg($client, $dbh);
    } else {
      $log->write("Got non pdw request via smtp from:".$remote_host);
    }

    # ensure the child exits correctly
    exit 0;
  }
  else {
    $log->write("Cannot fork process for new client connection: $!");
  }
}
## main ##
